<?php
// File: /fuelpos/admin/stations.php
// Gestión de estaciones (compatible con PDO o MySQLi).
// Requiere init.php que provea la conexión (ej. $pdo o $mysqli / $db / $dbh / $conn).
require __DIR__ . '/../init.php';
require_login();

if (!function_exists('esc')) {
    function esc($s){ return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}

/*
 Detectar conexión disponible en init.php.
 Intentamos detectar en este orden:
  - $pdo (PDO)
  - $db / $dbh (PDO or mysqli)
  - $mysqli / $conn (mysqli)
Si no se detecta ninguna, mostramos la ayuda con SQL (como antes).
*/
$pdo_conn = null;
$mysqli_conn = null;

foreach (['pdo','$pdo','db','dbh','$db','$dbh'] as $n) {
    // ignore these, we'll check actual vars below
}

if (isset($pdo) && $pdo instanceof PDO) {
    $pdo_conn = $pdo;
} elseif (isset($db) && $db instanceof PDO) {
    $pdo_conn = $db;
} elseif (isset($dbh) && $dbh instanceof PDO) {
    $pdo_conn = $dbh;
} elseif (isset($conn) && $conn instanceof PDO) {
    $pdo_conn = $conn;
}

// mysqli detection
if (!$pdo_conn) {
    if (isset($mysqli) && $mysqli instanceof mysqli) {
        $mysqli_conn = $mysqli;
    } elseif (isset($db) && $db instanceof mysqli) {
        $mysqli_conn = $db;
    } elseif (isset($dbh) && $dbh instanceof mysqli) {
        $mysqli_conn = $dbh;
    } elseif (isset($conn) && $conn instanceof mysqli) {
        $mysqli_conn = $conn;
    }
}

// If still no connection, show helpful message and SQL
if (!$pdo_conn && !$mysqli_conn) {
    ?>
    <!doctype html>
    <html lang="es">
    <head>
      <meta charset="utf-8">
      <title>Gestionar estación - Error configuración BD</title>
      <meta name="viewport" content="width=device-width,initial-scale=1">
      <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    </head>
    <body class="bg-light">
    <div class="container py-4">
      <h3>Gestionar estación</h3>
      <div class="alert alert-warning">
        No se detectó una conexión válida en init.php. Por favor expone tu conexión PDO como <code>$pdo</code> o expón un objeto MySQLi como <code>$mysqli</code> (o usa nombres <code>$db</code>, <code>$dbh</code>, <code>$conn</code> que detecta este script).
      </div>

      <h5>SQL para crear la tabla "stations"</h5>
      <pre><code>CREATE TABLE stations (
  id INT AUTO_INCREMENT PRIMARY KEY,
  nombre VARCHAR(255) NOT NULL,
  direccion VARCHAR(255) DEFAULT NULL,
  telefono VARCHAR(50) DEFAULT NULL,
  logo VARCHAR(255) DEFAULT NULL,
  rnc VARCHAR(50) DEFAULT NULL,
  mensaje1 TEXT,
  mensaje2 TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;</code></pre>

      <p>Ejemplo: en <code>init.php</code> añade/expone:</p>
      <pre><code>// PDO
$pdo = new PDO('mysql:host=localhost;dbname=tu_db;charset=utf8mb4','usuario','pass', [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);

// o MySQLi (orientado a objetos)
$mysqli = new mysqli('localhost','usuario','pass','tu_db');
if ($mysqli->connect_errno) { die('MySQLi connect error: ' . $mysqli->connect_error); }
</code></pre>

      <p>Si prefieres que adapte este archivo para la forma exacta de tu init.php, pega aquí el fragmento de init.php que crea la conexión y lo adapto.</p>
    </div>
    </body>
    </html>
    <?php
    exit;
}

/* ---------- Funciones de abstracción DB (PDO o MySQLi) ---------- */

/**
 * db_fetch_all($sql, $params)
 * Devuelve array asociativo de filas.
 */
function db_fetch_all($sql, $params = []) {
    global $pdo_conn, $mysqli_conn;
    if ($pdo_conn) {
        $stmt = $pdo_conn->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    // mysqli
    $stmt = $mysqli_conn->prepare($sql);
    if ($params && count($params)) {
        // bind all as strings
        $types = str_repeat('s', count($params));
        $refs = [];
        foreach ($params as $k => $v) $refs[] = &$params[$k];
        array_unshift($refs, $types);
        call_user_func_array([$stmt, 'bind_param'], $refs);
    }
    $stmt->execute();
    if (method_exists($stmt, 'get_result')) {
        $res = $stmt->get_result();
        return $res->fetch_all(MYSQLI_ASSOC);
    }
    // Fallback: bind_result
    $meta = $stmt->result_metadata();
    $fields = [];
    $row = [];
    while ($f = $meta->fetch_field()) {
        $fields[] = $f->name;
        $row[$f->name] = null;
        $bindNames[] = &$row[$f->name];
    }
    if (!empty($bindNames)) {
        call_user_func_array([$stmt, 'bind_result'], $bindNames);
        $out = [];
        while ($stmt->fetch()) {
            $r = [];
            foreach ($row as $k => $v) $r[$k] = $v;
            $out[] = $r;
        }
        return $out;
    }
    return [];
}

/**
 * db_fetch_one($sql, $params)
 */
function db_fetch_one($sql, $params = []) {
    $rows = db_fetch_all($sql, $params);
    return $rows[0] ?? null;
}

/**
 * db_execute($sql, $params)
 * Ejecuta INSERT/UPDATE/DELETE.
 * Devuelve ['ok'=>bool, 'lastInsertId'=>.., 'affected'=>..]
 */
function db_execute($sql, $params = []) {
    global $pdo_conn, $mysqli_conn;
    if ($pdo_conn) {
        $stmt = $pdo_conn->prepare($sql);
        $ok = $stmt->execute($params);
        $last = null;
        if ($ok) {
            try { $last = $pdo_conn->lastInsertId(); } catch(Exception $e) { $last = null; }
        }
        return ['ok' => (bool)$ok, 'lastInsertId' => $last, 'affected' => $stmt->rowCount()];
    }
    // mysqli
    $stmt = $mysqli_conn->prepare($sql);
    if ($params && count($params)) {
        $types = str_repeat('s', count($params));
        $refs = [];
        foreach ($params as $k => $v) $refs[] = &$params[$k];
        array_unshift($refs, $types);
        call_user_func_array([$stmt, 'bind_param'], $refs);
    }
    $ok = $stmt->execute();
    $affected = $stmt->affected_rows;
    $last = $mysqli_conn->insert_id;
    return ['ok' => (bool)$ok, 'lastInsertId' => $last, 'affected' => $affected];
}

/* ---------- Upload dir ---------- */
$uploadDir = __DIR__ . '/../assets/uploads';
$uploadWebPath = site('assets/uploads'); // ejemplo: /fuelpos/assets/uploads
if (!is_dir($uploadDir)) {
    @mkdir($uploadDir, 0755, true);
}

/* ---------- Manejo de acciones (create/update/delete) ---------- */
$action = $_POST['action'] ?? $_GET['action'] ?? '';
$messages = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($action === 'create' || $action === 'update') {
        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        $nombre = trim($_POST['nombre'] ?? '');
        $direccion = trim($_POST['direccion'] ?? '');
        $telefono = trim($_POST['telefono'] ?? '');
        $rnc = trim($_POST['rnc'] ?? '');
        $mensaje1 = trim($_POST['mensaje1'] ?? '');
        $mensaje2 = trim($_POST['mensaje2'] ?? '');
        $logoPath = null;

        // Handle file upload safely
        if (!empty($_FILES['logo']['name'])) {
            $f = $_FILES['logo'];
            if ($f['error'] === UPLOAD_ERR_OK) {
                $ext = strtolower(pathinfo($f['name'], PATHINFO_EXTENSION));
                // validate extension
                $allowed = ['png','jpg','jpeg','gif','webp'];
                if (!in_array($ext, $allowed)) {
                    $messages[] = ['type' => 'danger', 'text' => 'Formato de imagen no permitido. Usa png/jpg/gif/webp.'];
                } else {
                    $safe = 'station_logo_' . ($id ?: time()) . '_' . bin2hex(random_bytes(6)) . '.' . $ext;
                    $dest = $uploadDir . '/' . $safe;
                    if (move_uploaded_file($f['tmp_name'], $dest)) {
                        $logoPath = rtrim($uploadWebPath, '/') . '/' . $safe;
                    } else {
                        $messages[] = ['type' => 'danger', 'text' => 'No se pudo mover el archivo de logo. Verifica permisos en assets/uploads.'];
                    }
                }
            } else {
                $messages[] = ['type' => 'warning', 'text' => 'Error al subir el archivo (código ' . intval($f['error']) . ').'];
            }
        }

        if ($action === 'create') {
            $sql = "INSERT INTO stations (nombre, direccion, telefono, logo, rnc, mensaje1, mensaje2) VALUES (:nombre, :direccion, :telefono, :logo, :rnc, :mensaje1, :mensaje2)";
            if ($pdo_conn) {
                $res = db_execute($sql, [
                    ':nombre' => $nombre,
                    ':direccion' => $direccion ?: null,
                    ':telefono' => $telefono ?: null,
                    ':logo' => $logoPath,
                    ':rnc' => $rnc ?: null,
                    ':mensaje1' => $mensaje1 ?: null,
                    ':mensaje2' => $mensaje2 ?: null,
                ]);
            } else {
                // mysqli: convert named params to positional
                $sqlm = "INSERT INTO stations (nombre, direccion, telefono, logo, rnc, mensaje1, mensaje2) VALUES (?,?,?,?,?,?,?)";
                $res = db_execute($sqlm, [$nombre, $direccion ?: null, $telefono ?: null, $logoPath, $rnc ?: null, $mensaje1 ?: null, $mensaje2 ?: null]);
            }
            if ($res['ok']) $messages[] = ['type' => 'success', 'text' => 'Estación creada correctamente.'];
            else $messages[] = ['type' => 'danger', 'text' => 'Error al crear la estación.'];
        } elseif ($action === 'update' && $id) {
            if ($logoPath) {
                if ($pdo_conn) {
                    $sql = "UPDATE stations SET nombre=:nombre, direccion=:direccion, telefono=:telefono, logo=:logo, rnc=:rnc, mensaje1=:mensaje1, mensaje2=:mensaje2 WHERE id=:id";
                    $params = [':nombre'=>$nombre, ':direccion'=>$direccion?:null, ':telefono'=>$telefono?:null, ':logo'=>$logoPath, ':rnc'=>$rnc?:null, ':mensaje1'=>$mensaje1?:null, ':mensaje2'=>$mensaje2?:null, ':id'=>$id];
                    $res = db_execute($sql, $params);
                } else {
                    $sql = "UPDATE stations SET nombre=?, direccion=?, telefono=?, logo=?, rnc=?, mensaje1=?, mensaje2=? WHERE id=?";
                    $res = db_execute($sql, [$nombre, $direccion?:null, $telefono?:null, $logoPath, $rnc?:null, $mensaje1?:null, $mensaje2?:null, $id]);
                }
            } else {
                if ($pdo_conn) {
                    $sql = "UPDATE stations SET nombre=:nombre, direccion=:direccion, telefono=:telefono, rnc=:rnc, mensaje1=:mensaje1, mensaje2=:mensaje2 WHERE id=:id";
                    $params = [':nombre'=>$nombre, ':direccion'=>$direccion?:null, ':telefono'=>$telefono?:null, ':rnc'=>$rnc?:null, ':mensaje1'=>$mensaje1?:null, ':mensaje2'=>$mensaje2?:null, ':id'=>$id];
                    $res = db_execute($sql, $params);
                } else {
                    $sql = "UPDATE stations SET nombre=?, direccion=?, telefono=?, rnc=?, mensaje1=?, mensaje2=? WHERE id=?";
                    $res = db_execute($sql, [$nombre, $direccion?:null, $telefono?:null, $rnc?:null, $mensaje1?:null, $mensaje2?:null, $id]);
                }
            }
            if ($res['ok']) $messages[] = ['type' => 'success', 'text' => 'Estación actualizada correctamente.'];
            else $messages[] = ['type' => 'danger', 'text' => 'Error al actualizar la estación.'];
        }
    } elseif ($action === 'delete' && !empty($_POST['id'])) {
        $id = intval($_POST['id']);
        // obtener logo para borrar archivo (si aplica)
        $row = db_fetch_one("SELECT logo FROM stations WHERE id = " . intval($id));
        if ($row && !empty($row['logo'])) {
            $file = __DIR__ . '/../' . ltrim($row['logo'], '/');
            if (is_file($file)) @unlink($file);
        }
        if ($pdo_conn) {
            $res = db_execute("DELETE FROM stations WHERE id = :id", [':id'=>$id]);
        } else {
            $res = db_execute("DELETE FROM stations WHERE id = ?", [$id]);
        }
        if ($res['ok']) $messages[] = ['type'=>'success', 'text'=>'Estación eliminada.'];
        else $messages[] = ['type'=>'danger', 'text'=>'Error al eliminar estación.'];
    }
}

/* ---------- Obtener estaciones ---------- */
if ($pdo_conn) {
    $stations = db_fetch_all("SELECT * FROM stations ORDER BY id ASC");
} else {
    $stations = db_fetch_all("SELECT * FROM stations ORDER BY id ASC");
}

?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Gestionar Estación</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .logo-thumb { max-width:120px; max-height:60px; object-fit:contain; }
    .station-row { border-bottom:1px solid #f1f1f1; padding:12px 0; }
  </style>
</head>
<body class="bg-light">
<nav class="navbar navbar-light bg-white shadow-sm mb-4">
  <div class="container">
    <a class="navbar-brand" href="<?= site('admin/index.php') ?>">Admin</a>
    <div class="ms-auto">
      <a class="btn btn-outline-secondary btn-sm" href="<?= site('admin/index.php') ?>">Volver</a>
    </div>
  </div>
</nav>

<div class="container">
  <h3>Gestionar Estación</h3>

  <?php foreach ($messages as $m): ?>
    <div class="alert alert-<?= esc($m['type']) ?>"><?= esc($m['text']) ?></div>
  <?php endforeach; ?>

  <div class="mb-4">
    <h5>Crear nueva estación</h5>
    <form method="post" enctype="multipart/form-data" class="row g-2">
      <input type="hidden" name="action" value="create">
      <div class="col-md-6">
        <label class="form-label">Nombre</label>
        <input class="form-control" name="nombre" required>
      </div>
      <div class="col-md-6">
        <label class="form-label">Teléfono</label>
        <input class="form-control" name="telefono">
      </div>
      <div class="col-md-6">
        <label class="form-label">Dirección</label>
        <input class="form-control" name="direccion">
      </div>
      <div class="col-md-6">
        <label class="form-label">RNC</label>
        <input class="form-control" name="rnc">
      </div>
      <div class="col-md-6">
        <label class="form-label">Logo (imagen)</label>
        <input type="file" class="form-control" name="logo" accept="image/*">
      </div>
      <div class="col-12">
        <label class="form-label">Mensaje 1 (para ticket)</label>
        <input class="form-control" name="mensaje1">
      </div>
      <div class="col-12">
        <label class="form-label">Mensaje 2 (para ticket)</label>
        <input class="form-control" name="mensaje2">
      </div>
      <div class="col-12">
        <button class="btn btn-success">Crear estación</button>
      </div>
    </form>
  </div>

  <hr>

  <h5>Estaciones existentes</h5>
  <?php if (empty($stations)): ?>
    <div class="alert alert-info">No hay estaciones registradas.</div>
  <?php else: ?>
    <?php foreach ($stations as $st): ?>
      <div class="station-row row align-items-center">
        <div class="col-md-3">
          <?php if (!empty($st['logo'])): ?>
            <img src="<?= esc($st['logo']) ?>" class="logo-thumb img-thumbnail" alt="logo">
          <?php else: ?>
            <div class="text-muted small">Sin logo</div>
          <?php endif; ?>
        </div>
        <div class="col-md-6">
          <strong><?= esc($st['nombre']) ?></strong><br>
          <small class="text-muted"><?= esc($st['direccion']) ?> <?= $st['telefono'] ? '• Tel: '.esc($st['telefono']) : '' ?></small>
        </div>
        <div class="col-md-3 text-end">
          <a class="btn btn-sm btn-outline-primary" href="?edit=<?= intval($st['id']) ?>">Editar</a>
          <form method="post" style="display:inline" onsubmit="return confirm('Eliminar estación?');">
            <input type="hidden" name="action" value="delete">
            <input type="hidden" name="id" value="<?= intval($st['id']) ?>">
            <button class="btn btn-sm btn-outline-danger">Eliminar</button>
          </form>
        </div>

        <?php if (isset($_GET['edit']) && intval($_GET['edit']) === intval($st['id'])): ?>
          <div class="col-12 mt-3">
            <div class="card p-3 bg-white">
              <form method="post" enctype="multipart/form-data" class="row g-2">
                <input type="hidden" name="action" value="update">
                <input type="hidden" name="id" value="<?= intval($st['id']) ?>">
                <div class="col-md-6">
                  <label class="form-label">Nombre</label>
                  <input class="form-control" name="nombre" required value="<?= esc($st['nombre']) ?>">
                </div>
                <div class="col-md-6">
                  <label class="form-label">Teléfono</label>
                  <input class="form-control" name="telefono" value="<?= esc($st['telefono']) ?>">
                </div>
                <div class="col-md-6">
                  <label class="form-label">Dirección</label>
                  <input class="form-control" name="direccion" value="<?= esc($st['direccion']) ?>">
                </div>
                <div class="col-md-6">
                  <label class="form-label">RNC</label>
                  <input class="form-control" name="rnc" value="<?= esc($st['rnc']) ?>">
                </div>
                <div class="col-md-6">
                  <label class="form-label">Logo (subir nuevo para reemplazar)</label>
                  <input type="file" class="form-control" name="logo" accept="image/*">
                </div>
                <div class="col-12">
                  <label class="form-label">Mensaje 1</label>
                  <input class="form-control" name="mensaje1" value="<?= esc($st['mensaje1']) ?>">
                </div>
                <div class="col-12">
                  <label class="form-label">Mensaje 2</label>
                  <input class="form-control" name="mensaje2" value="<?= esc($st['mensaje2']) ?>">
                </div>
                <div class="col-12 text-end">
                  <button class="btn btn-primary">Guardar</button>
                  <a class="btn btn-light" href="<?= site('admin/stations.php') ?>">Cancelar</a>
                </div>
              </form>
            </div>
          </div>
        <?php endif; ?>

      </div>
    <?php endforeach; ?>
  <?php endif; ?>

</div>
</body>
</html>