<?php
// admin/tanques.php - Gestión de tanques y compras de combustible

require_once __DIR__ . '/../init.php';
require_login();

$pdo = $GLOBALS['pdo'] ?? null;
if (!$pdo && isset($DB) && $DB instanceof PDO) $pdo = $DB;
if (!$pdo && isset($db) && $db instanceof PDO) $pdo = $db;
if (!$pdo && isset($conn) && $conn instanceof PDO) $pdo = $conn;

if (!$pdo || !($pdo instanceof PDO)) {
    die("Error: no se encontró una conexión PDO. Revisa init.php.");
}

function fetch_all($sql, $params = []) {
    global $pdo;
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}
function fetch_one($sql, $params = []) {
    global $pdo;
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}
function execute_stmt($sql, $params = []) {
    global $pdo;
    $stmt = $pdo->prepare($sql);
    return $stmt->execute($params);
}
function redirect($url) {
    header("Location: $url");
    exit;
}

$action = $_GET['action'] ?? 'list';
$errors = [];

// POST handling: create/update/delete tanques, compras y ajustes de menos
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $form_action = $_POST['form_action'] ?? '';

    // 1. Crear/editar/eliminar tanques
    if (in_array($form_action, ['create', 'update', 'delete'])) {
        $producto_id = isset($_POST['producto_id']) ? (int)$_POST['producto_id'] : 0;
        $capacidad = isset($_POST['capacidad']) ? (float)$_POST['capacidad'] : 0.0;
        $vendido = isset($_POST['vendido']) ? (float)$_POST['vendido'] : 0.0;
        $stock_inicial = isset($_POST['stock_inicial']) ? (float)$_POST['stock_inicial'] : 0.0;

        // Validaciones básicas
        if ($producto_id <= 0) $errors[] = "Producto es requerido.";
        if ($capacidad < 0) $errors[] = "Capacidad debe ser >= 0.";
        if ($vendido < 0) $errors[] = "Vendido debe ser >= 0.";
        if ($stock_inicial < 0) $errors[] = "Stock inicial debe ser >= 0.";

        if (empty($errors)) {
            try {
                if ($form_action === 'create') {
                    $disponible = $stock_inicial - $vendido;
                    execute_stmt("INSERT INTO tanques (producto_id, capacidad, vendido, disponible, stock_inicial, created_at, updated_at) VALUES (?, ?, ?, ?, ?, NOW(), NOW())",
                                 [$producto_id, $capacidad, $vendido, $disponible, $stock_inicial]);
                    redirect('tanques.php');
                } elseif ($form_action === 'update' && isset($_POST['id'])) {
                    $id = (int)$_POST['id'];
                    execute_stmt("UPDATE tanques SET producto_id = ?, capacidad = ?, vendido = ?, updated_at = NOW() WHERE id = ?",
                                 [$producto_id, $capacidad, $vendido, $id]);
                    redirect('tanques.php');
                } elseif ($form_action === 'delete' && isset($_POST['id'])) {
                    $id = (int)$_POST['id'];
                    execute_stmt("DELETE FROM tanques WHERE id = ?", [$id]);
                    redirect('tanques.php');
                } else {
                    $errors[] = "Acción inválida.";
                }
            } catch (Exception $e) {
                $errors[] = "Error en la operación: " . $e->getMessage();
            }
        }
    }

    // 2. Registrar compra de combustible
    if ($form_action === 'add_compra') {
        $tanque_id = (int)($_POST['tanque_id'] ?? 0);
        $cantidad = (float)($_POST['cantidad'] ?? 0);
        $proveedor = trim($_POST['proveedor'] ?? '');
        $observaciones = trim($_POST['observaciones'] ?? '');
        if ($tanque_id <= 0) $errors[] = "Tanque inválido.";
        if ($cantidad <= 0) $errors[] = "Cantidad debe ser mayor a cero.";
        if (empty($errors)) {
            try {
                execute_stmt("INSERT INTO compras_tanques (tanque_id, cantidad, proveedor, observaciones, fecha) VALUES (?, ?, ?, ?, NOW())",
                    [$tanque_id, $cantidad, $proveedor, $observaciones]);
                execute_stmt("UPDATE tanques SET disponible = disponible + ? WHERE id = ?", [$cantidad, $tanque_id]);
                redirect('tanques.php');
            } catch (Exception $e) {
                $errors[] = "Error al registrar la compra: " . $e->getMessage();
            }
        }
    }

    // 3. Ajuste de menos (evaporación, corrección)
    if ($form_action === 'ajuste_menos') {
        $tanque_id = (int)($_POST['tanque_id'] ?? 0);
        $cantidad = (float)($_POST['cantidad'] ?? 0);
        $motivo = trim($_POST['motivo'] ?? '');
        if ($tanque_id <= 0) $errors[] = "Tanque inválido.";
        if ($cantidad >= 0) $errors[] = "El ajuste debe ser un valor negativo (menor a cero).";
        if (empty($motivo)) $errors[] = "Debe indicar un motivo.";
        if (empty($errors)) {
            try {
                // Puedes guardar ajustes en la misma tabla de compras o en otra. Aquí en compras_tanques como negativo.
                execute_stmt("INSERT INTO compras_tanques (tanque_id, cantidad, proveedor, observaciones, fecha) VALUES (?, ?, ?, ?, NOW())",
                    [$tanque_id, $cantidad, 'Ajuste de menos', $motivo]);
                execute_stmt("UPDATE tanques SET disponible = disponible + ? WHERE id = ?", [$cantidad, $tanque_id]);
                redirect('tanques.php');
            } catch (Exception $e) {
                $errors[] = "Error al registrar el ajuste: " . $e->getMessage();
            }
        }
    }
}

// -----------------
// Cargar productos
// -----------------
$productos = [];
$productos_error = '';
try {
    $tbl = fetch_all("SHOW TABLES LIKE 'products'");
    if (empty($tbl)) {
        $productos_error = "No se encontró la tabla 'products'.";
    } else {
        $productos = fetch_all("SELECT id, name AS display_name FROM products ORDER BY name ASC");
    }
} catch (Exception $e) {
    $productos = [];
    $productos_error = $e->getMessage();
}

// -----------------
// Listar tanques
// -----------------
$tanques = [];
$tanques_table_missing = false;
$list_error = '';
try {
    $tables = fetch_all("SHOW TABLES LIKE 'tanques'");
    if (empty($tables)) {
        $tanques_table_missing = true;
        $list_error = "La tabla 'tanques' no existe. Ejecuta la migración SQL para crearla.";
    } else {
        $tanques = fetch_all("SELECT t.*, p.name AS producto_nombre FROM tanques t LEFT JOIN products p ON p.id = t.producto_id ORDER BY t.id DESC");
    }
} catch (Exception $e) {
    $tanques = [];
    $list_error = $e->getMessage();
    $tanques_table_missing = true;
}

// Determinar si se permite crear (tabla existe y hay productos)
$canCreate = !$tanques_table_missing && !empty($productos);

?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Gestión de Tanques</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
<div class="container py-4">
  <h1>Gestión de Tanques</h1>
  <p class="mb-3">
    <a class="btn btn-sm btn-outline-secondary" href="<?= site('admin/index.php') ?>">Volver</a>
    <?php if ($canCreate): ?>
      <a class="btn btn-sm btn-primary" href="tanques.php?action=create">Nuevo Tanque</a>
      <a class="btn btn-sm btn-success" href="tanques.php?action=reportes">Reportes de compras</a>
    <?php endif; ?>
  </p>

  <?php if (!empty($productos_error)): ?>
    <div class="alert alert-warning">Productos: <?= htmlspecialchars($productos_error) ?></div>
  <?php endif; ?>

  <?php if (!empty($list_error)): ?>
    <div class="alert alert-danger"><?= htmlspecialchars($list_error) ?></div>
  <?php endif; ?>

  <?php if ($tanques_table_missing): ?>
    <div class="alert alert-info">
      La tabla <strong>tanques</strong> no existe. Ejecuta el archivo SQL incluido en el repo para crearla.
    </div>
  <?php endif; ?>

  <div class="table-responsive">
    <table class="table table-striped table-bordered align-middle">
      <thead>
        <tr>
          <th>ID</th>
          <th>Producto</th>
          <th>Capacidad</th>
          <th>Stock inicial</th>
          <th>Vendido</th>
          <th>Disponible</th>
          <th>Acciones</th>
        </tr>
      </thead>
      <tbody>
        <?php if (empty($tanques)): ?>
          <tr><td colspan="7" class="text-center">No hay registros</td></tr>
        <?php else: ?>
          <?php foreach ($tanques as $t): ?>
            <tr>
              <td><?= htmlspecialchars($t['id']) ?></td>
              <td><?= htmlspecialchars($t['producto_nombre'] ?? 'N/A') ?></td>
              <td><?= htmlspecialchars($t['capacidad']) ?></td>
              <td><?= htmlspecialchars($t['stock_inicial']) ?></td>
              <td><?= htmlspecialchars($t['vendido']) ?></td>
              <td><?= htmlspecialchars($t['disponible']) ?></td>
              <td>
                <a class="btn btn-sm btn-outline-primary" href="tanques.php?action=edit&id=<?= $t['id'] ?>">Editar</a>
                <a class="btn btn-sm btn-outline-danger" href="tanques.php?action=delete&id=<?= $t['id'] ?>">Eliminar</a>
                <a class="btn btn-sm btn-success" href="tanques.php?action=add_compra&id=<?= $t['id'] ?>">Agregar compra</a>
                <a class="btn btn-sm btn-danger" href="tanques.php?action=ajuste_menos&id=<?= $t['id'] ?>">Ajuste de menos</a>
              </td>
            </tr>
          <?php endforeach; ?>
        <?php endif; ?>
      </tbody>
    </table>
  </div>
</div>

<?php if ($action === 'create'): 
    if (!$canCreate): ?>
      <div class="container py-3">
        <div class="alert alert-warning">No puede crear tanques hasta que exista la tabla <strong>tanques</strong> y al menos un producto en la tabla <strong>products</strong>.</div>
      </div>
    <?php else: ?>
      <div class="container py-4">
        <h2>Crear Tanque</h2>
        <?php if (!empty($errors)): ?>
          <div class="alert alert-danger"><ul><?php foreach ($errors as $er) echo "<li>" . htmlspecialchars($er) . "</li>"; ?></ul></div>
        <?php endif; ?>
        <form method="post" action="tanques.php">
          <input type="hidden" name="form_action" value="create">
          <div class="mb-3">
            <label class="form-label">Producto</label>
            <select name="producto_id" class="form-select" required>
              <option value="">-- Seleccione --</option>
              <?php foreach ($productos as $p): ?>
                <option value="<?= $p['id'] ?>"><?= htmlspecialchars($p['display_name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="mb-3">
            <label class="form-label">Capacidad</label>
            <input type="number" step="0.001" name="capacidad" class="form-control" value="0.000" required>
          </div>
          <div class="mb-3">
            <label class="form-label">Stock inicial</label>
            <input type="number" step="0.001" name="stock_inicial" class="form-control" value="0.000" required>
            <div class="form-text">Asignar un valor inicial antes de iniciar ventas en POS.</div>
          </div>
          <div class="mb-3">
            <label class="form-label">Vendido</label>
            <input type="number" step="0.001" name="vendido" class="form-control" value="0.000" required>
          </div>
          <button class="btn btn-primary" type="submit">Guardar Tanque</button>
        </form>
      </div>
    <?php endif; 
endif; ?>

<?php if ($action === 'edit' && isset($_GET['id'])):
    $id = (int)$_GET['id'];
    $t = fetch_one("SELECT * FROM tanques WHERE id = ?", [$id]);
    if (!$t) { echo "<div class='container py-4'><div class='alert alert-danger'>Tanque no encontrado.</div></div>"; exit; }
?>
  <div class="container py-4">
    <h2>Editar Tanque #<?= htmlspecialchars($t['id']) ?></h2>
    <?php if (!empty($errors)): ?>
      <div class="alert alert-danger"><ul><?php foreach ($errors as $er) echo "<li>" . htmlspecialchars($er) . "</li>"; ?></ul></div>
    <?php endif; ?>
    <form method="post" action="tanques.php">
      <input type="hidden" name="form_action" value="update">
      <input type="hidden" name="id" value="<?= htmlspecialchars($t['id']) ?>">
      <div class="mb-3">
        <label class="form-label">Producto</label>
        <select name="producto_id" class="form-select" required>
          <option value="">-- Seleccione --</option>
          <?php foreach ($productos as $p): ?>
            <option value="<?= $p['id'] ?>" <?= $p['id'] == $t['producto_id'] ? 'selected' : '' ?>><?= htmlspecialchars($p['display_name']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="mb-3">
        <label class="form-label">Capacidad</label>
        <input type="number" step="0.001" name="capacidad" class="form-control" value="<?= htmlspecialchars($t['capacidad']) ?>" required>
      </div>
      <div class="mb-3">
        <label class="form-label">Vendido</label>
        <input type="number" step="0.001" name="vendido" class="form-control" value="<?= htmlspecialchars($t['vendido']) ?>" required>
      </div>
      <button class="btn btn-primary" type="submit">Actualizar Tanque</button>
    </form>
  </div>
<?php endif; ?>

<?php if ($action === 'add_compra' && isset($_GET['id'])):
    $id = (int)$_GET['id'];
    $t = fetch_one("SELECT t.*, p.name AS producto_nombre FROM tanques t LEFT JOIN products p ON p.id = t.producto_id WHERE t.id = ?", [$id]);
    if (!$t) { echo "<div class='container py-4'><div class='alert alert-danger'>Tanque no encontrado.</div></div>"; exit; }
?>
  <div class="container py-4">
    <h2>Agregar compra a tanque: <?= htmlspecialchars($t['producto_nombre']) ?></h2>
    <?php if (!empty($errors)): ?>
      <div class="alert alert-danger"><ul><?php foreach ($errors as $er) echo "<li>" . htmlspecialchars($er) . "</li>"; ?></ul></div>
    <?php endif; ?>
    <form method="post" action="tanques.php">
      <input type="hidden" name="form_action" value="add_compra">
      <input type="hidden" name="tanque_id" value="<?= htmlspecialchars($t['id']) ?>">
      <div class="mb-3">
        <label class="form-label">Cantidad comprada</label>
        <input type="number" step="0.001" name="cantidad" class="form-control" min="0.001" required>
      </div>
      <div class="mb-3">
        <label class="form-label">Proveedor</label>
        <input type="text" name="proveedor" class="form-control">
      </div>
      <div class="mb-3">
        <label class="form-label">Observaciones</label>
        <textarea name="observaciones" class="form-control"></textarea>
      </div>
      <button class="btn btn-success" type="submit">Agregar compra</button>
      <a class="btn btn-secondary" href="tanques.php">Cancelar</a>
    </form>
  </div>
<?php endif; ?>

<?php if ($action === 'ajuste_menos' && isset($_GET['id'])):
    $id = (int)$_GET['id'];
    $t = fetch_one("SELECT t.*, p.name AS producto_nombre FROM tanques t LEFT JOIN products p ON p.id = t.producto_id WHERE t.id = ?", [$id]);
    if (!$t) { echo "<div class='container py-4'><div class='alert alert-danger'>Tanque no encontrado.</div></div>"; exit; }
?>
  <div class="container py-4">
    <h2>Ajuste de menos a tanque: <?= htmlspecialchars($t['producto_nombre']) ?></h2>
    <div class="alert alert-info">Use este formulario para descontar del inventario por evaporación, ajuste interno etc. El valor debe ser negativo.</div>
    <?php if (!empty($errors)): ?>
      <div class="alert alert-danger"><ul><?php foreach ($errors as $er) echo "<li>" . htmlspecialchars($er) . "</li>"; ?></ul></div>
    <?php endif; ?>
    <form method="post" action="tanques.php">
      <input type="hidden" name="form_action" value="ajuste_menos">
      <input type="hidden" name="tanque_id" value="<?= htmlspecialchars($t['id']) ?>">
      <div class="mb-3">
        <label class="form-label">Cantidad a descontar (debe ser negativo)</label>
        <input type="number" step="0.001" name="cantidad" class="form-control" max="-0.001" required>
        <div class="form-text">Ejemplo: -10.500 (solo valores negativos)</div>
      </div>
      <div class="mb-3">
        <label class="form-label">Motivo del ajuste</label>
        <textarea name="motivo" class="form-control" required placeholder="Indique la razón del ajuste, ejemplo: evaporación, faltante, ajuste inventario"></textarea>
      </div>
      <button class="btn btn-warning" type="submit">Guardar ajuste de menos</button>
      <a class="btn btn-secondary" href="tanques.php">Cancelar</a>
    </form>
  </div>
<?php endif; ?>

<?php if ($action === 'delete' && isset($_GET['id'])):
    $id = (int)$_GET['id'];
    $t = fetch_one("SELECT t.*, p.name AS producto_nombre FROM tanques t LEFT JOIN products p ON p.id = t.producto_id WHERE t.id = ?", [$id]);
    if (!$t) { echo "<div class='container py-4'><div class='alert alert-danger'>Tanque no encontrado.</div></div>"; exit; }
?>
  <div class="container py-4">
    <h2>Eliminar Tanque #<?= htmlspecialchars($t['id']) ?></h2>
    <p>¿Seguro que quieres eliminar el tanque para el producto: <strong><?= htmlspecialchars($t['producto_nombre'] ?? 'N/A') ?></strong> ?</p>
    <form method="post" action="tanques.php">
      <input type="hidden" name="form_action" value="delete">
      <input type="hidden" name="id" value="<?= htmlspecialchars($t['id']) ?>">
      <button class="btn btn-danger" type="submit">Sí, eliminar</button>
      <a class="btn btn-secondary" href="tanques.php">Cancelar</a>
    </form>
  </div>
<?php endif; ?>

<?php
// PANTALLA DE REPORTES DE COMPRAS
if ($action === 'reportes'):

// Filtros
$filtro_fecha_ini = $_GET['fecha_ini'] ?? '';
$filtro_fecha_fin = $_GET['fecha_fin'] ?? '';
$filtro_producto = isset($_GET['producto_id']) && is_numeric($_GET['producto_id']) ? (int)$_GET['producto_id'] : '';

$where = [];
$params = [];

if ($filtro_fecha_ini) {
    $where[] = "c.fecha >= ?";
    $params[] = $filtro_fecha_ini . " 00:00:00";
}
if ($filtro_fecha_fin) {
    $where[] = "c.fecha <= ?";
    $params[] = $filtro_fecha_fin . " 23:59:59";
}
if ($filtro_producto) {
    $where[] = "t.producto_id = ?";
    $params[] = $filtro_producto;
}

$where_clause = $where ? "WHERE " . implode(" AND ", $where) : "";

$sql = "
    SELECT c.*, t.producto_id, p.name AS producto_nombre 
    FROM compras_tanques c
    JOIN tanques t ON t.id = c.tanque_id
    JOIN products p ON p.id = t.producto_id
    $where_clause
    ORDER BY c.fecha DESC, c.id DESC
";
$compras = fetch_all($sql, $params);
?>
<div class="container py-4">
  <h2>Reporte de Compras de Combustible</h2>
  <form class="row g-3 mb-4" method="get" action="tanques.php">
    <input type="hidden" name="action" value="reportes">
    <div class="col-md-3">
      <label class="form-label">Fecha desde</label>
      <input type="date" name="fecha_ini" class="form-control" value="<?= htmlspecialchars($filtro_fecha_ini) ?>">
    </div>
    <div class="col-md-3">
      <label class="form-label">Fecha hasta</label>
      <input type="date" name="fecha_fin" class="form-control" value="<?= htmlspecialchars($filtro_fecha_fin) ?>">
    </div>
    <div class="col-md-4">
      <label class="form-label">Tipo de combustible</label>
      <select name="producto_id" class="form-select">
        <option value="">-- Todos --</option>
        <?php foreach ($productos as $p): ?>
          <option value="<?= $p['id'] ?>" <?= ($p['id'] == $filtro_producto) ? 'selected' : '' ?>>
            <?= htmlspecialchars($p['display_name']) ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>
    <div class="col-md-2 d-flex align-items-end">
      <button class="btn btn-primary w-100" type="submit">Filtrar</button>
    </div>
  </form>
  <div class="table-responsive">
    <table class="table table-striped table-bordered align-middle">
      <thead>
        <tr>
          <th>Fecha</th>
          <th>Producto</th>
          <th>Cantidad</th>
          <th>Proveedor</th>
          <th>Observaciones</th>
        </tr>
      </thead>
      <tbody>
        <?php if (empty($compras)): ?>
          <tr><td colspan="5" class="text-center">No hay compras en este rango/filtro.</td></tr>
        <?php else: ?>
          <?php foreach ($compras as $c): ?>
            <tr>
              <td><?= htmlspecialchars($c['fecha']) ?></td>
              <td><?= htmlspecialchars($c['producto_nombre']) ?></td>
              <td><?= htmlspecialchars($c['cantidad']) ?></td>
              <td><?= htmlspecialchars($c['proveedor']) ?></td>
              <td><?= htmlspecialchars($c['observaciones']) ?></td>
            </tr>
          <?php endforeach; ?>
        <?php endif; ?>
      </tbody>
    </table>
  </div>
  <a class="btn btn-secondary mt-3" href="tanques.php">Volver</a>
</div>
<?php endif; ?>

</body>
</html>