<?php
require __DIR__ . '/../init.php';
require_login();
$pdo = pdo();

/**
 * admin/users.php - Gestión avanzada de usuarios/roles/permisos
 * - Diseño moderno con gradientes
 * - Sistema de permisos por archivos PHP
 * - Administrador tiene acceso total sin restricciones
 * - Auto-creación de tablas necesarias
 */

/* ------------- Helpers ------------- */
function esc($s){ return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }

function col_exists(PDO $pdo, string $table, string $col): bool {
    try {
        $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
        $st->execute([$table, $col]);
        return (int)$st->fetchColumn() > 0;
    } catch (Exception $e) {
        return false;
    }
}

function table_exists(PDO $pdo, string $table): bool {
    try {
        $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
        $st->execute([$table]);
        return (int)$st->fetchColumn() > 0;
    } catch (Exception $e) {
        return false;
    }
}

function first_existing_col(PDO $pdo, string $table, array $candidates): ?string {
    foreach ($candidates as $c) if (col_exists($pdo, $table, $c)) return $c;
    return null;
}

function flash($msg, $type='ok') {
    if ($type === 'ok') $_SESSION['flash_ok'] = $msg; else $_SESSION['flash_error'] = $msg;
}

// ✅ CREAR TABLAS SI NO EXISTEN
try {
    // Verificar y crear columna is_admin en roles
    if (!col_exists($pdo, 'roles', 'is_admin')) {
        $pdo->exec("ALTER TABLE roles ADD COLUMN is_admin TINYINT(1) DEFAULT 0 AFTER description");
    }
    
    // Verificar y crear tabla role_file_permissions
    if (!table_exists($pdo, 'role_file_permissions')) {
        $pdo->exec("
            CREATE TABLE role_file_permissions (
                id INT AUTO_INCREMENT PRIMARY KEY,
                role_id INT NOT NULL,
                file_path VARCHAR(255) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_role (role_id),
                UNIQUE KEY unique_role_file (role_id, file_path)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ");
    }
} catch (Exception $e) {
    error_log("Error creando estructuras de BD: " . $e->getMessage());
}

// ✅ OBTENER TODOS LOS ARCHIVOS PHP DEL SISTEMA
function get_php_files(): array {
    $baseDir = __DIR__ . '/../';
    $files = [];
    
    // Escanear directorios principales
    $directories = [
        '' => 'Raíz',
        'admin/' => 'Admin',
        'pos/' => 'POS',
        'ncf/' => 'Comprobantes',
    ];
    
    foreach ($directories as $dir => $label) {
        $fullPath = $baseDir . $dir;
        if (!is_dir($fullPath)) continue;
        
        $dirFiles = glob($fullPath . '*.php');
        if ($dirFiles === false) continue;
        
        foreach ($dirFiles as $file) {
            $filename = basename($file);
            // Excluir archivos de sistema
            if (in_array($filename, ['init.php', 'config.php', 'db.php'])) continue;
            
            $files[] = [
                'file' => $dir . $filename,
                'label' => $label . ' / ' . $filename,
                'group' => $label
            ];
        }
    }
    
    return $files;
}

/* detect optional columns */
$pwdCol = first_existing_col($pdo, 'users', ['password','password_hash','passwd','pass']);
$roleCol = first_existing_col($pdo, 'users', ['role','user_role','role_name']);
$fullNameCol = first_existing_col($pdo, 'users', ['full_name','name','display_name','fullname']);
$activeCol = first_existing_col($pdo, 'users', ['active','is_active','enabled']);

/* ------------- Actions ------------- */
$action = $_POST['action'] ?? $_GET['action'] ?? '';

try {
    // Save role
    if ($action === 'save_role') {
        $id = isset($_POST['role_id']) ? (int)$_POST['role_id'] : 0;
        $name = trim($_POST['name'] ?? '');
        $desc = trim($_POST['description'] ?? '');
        $is_admin = isset($_POST['is_admin']) ? 1 : 0;
        
        if ($name === '') { 
            flash('El nombre del rol es requerido.','error'); 
            header('Location:' . site('admin/users.php')); 
            exit; 
        }
        
        if ($id > 0) {
            $pdo->prepare("UPDATE roles SET name = ?, description = ?, is_admin = ? WHERE id = ?")->execute([$name, $desc, $is_admin, $id]);
            flash('Rol actualizado.');
        } else {
            $pdo->prepare("INSERT INTO roles (name, description, is_admin) VALUES (?, ?, ?)")->execute([$name, $desc, $is_admin]);
            flash('Rol creado.');
        }
        header('Location:' . site('admin/users.php')); 
        exit;
    }

    // Delete role
    if ($action === 'delete_role') {
        $id = isset($_POST['role_id']) ? (int)$_POST['role_id'] : 0;
        if ($id > 0) {
            $pdo->beginTransaction();
            $pdo->prepare("DELETE FROM role_permissions WHERE role_id = ?")->execute([$id]);
            $pdo->prepare("DELETE FROM role_file_permissions WHERE role_id = ?")->execute([$id]);
            $pdo->prepare("DELETE FROM user_roles WHERE role_id = ?")->execute([$id]);
            $pdo->prepare("DELETE FROM roles WHERE id = ?")->execute([$id]);
            $pdo->commit();
            flash('Rol eliminado.');
        } else flash('Rol inválido.','error');
        header('Location:' . site('admin/users.php')); 
        exit;
    }

    // ✅ GUARDAR PERMISOS DE ARCHIVOS
    if ($action === 'save_file_permissions') {
        $data = json_decode(file_get_contents('php://input'), true) ?: $_POST;
        $role_id = isset($data['role_id']) ? (int)$data['role_id'] : 0;
        $files = is_array($data['files']) ? $data['files'] : [];
        
        if ($role_id <= 0) { 
            header('Content-Type:application/json'); 
            echo json_encode(['ok'=>0,'msg'=>'role_id requerido']); 
            exit; 
        }
        
        $pdo->beginTransaction();
        $pdo->prepare("DELETE FROM role_file_permissions WHERE role_id = ?")->execute([$role_id]);
        
        if (!empty($files)) {
            $ins = $pdo->prepare("INSERT INTO role_file_permissions (role_id, file_path) VALUES (?, ?)");
            foreach ($files as $file) {
                if (is_string($file) && $file !== '') {
                    $ins->execute([$role_id, $file]);
                }
            }
        }
        
        $pdo->commit();
        header('Content-Type:application/json'); 
        echo json_encode(['ok'=>1,'msg'=>'Permisos de archivos actualizados']); 
        exit;
    }

    // Save permissions
    if ($action === 'save_permissions') {
        $data = json_decode(file_get_contents('php://input'), true) ?: $_POST;
        $role_id = isset($data['role_id']) ? (int)$data['role_id'] : 0;
        $perms = is_array($data['permissions']) ? $data['permissions'] : (isset($data['permissions']) ? [$data['permissions']] : []);
        $perms = array_values(array_filter(array_map(function($v){ return is_numeric($v) ? (int)$v : null; }, $perms)));
        
        if ($role_id <= 0) { 
            header('Content-Type:application/json'); 
            echo json_encode(['ok'=>0,'msg'=>'role_id requerido']); 
            exit; 
        }
        
        $validPerms = [];
        if (!empty($perms)) {
            $ph = implode(',', array_fill(0, count($perms), '?'));
            $st = $pdo->prepare("SELECT id FROM permissions WHERE id IN ($ph)");
            $st->execute($perms);
            $validPerms = array_map('intval', $st->fetchAll(PDO::FETCH_COLUMN,0));
        }
        
        $pdo->beginTransaction();
        $pdo->prepare("DELETE FROM role_permissions WHERE role_id = ?")->execute([$role_id]);
        
        if (!empty($validPerms)) {
            $ins = $pdo->prepare("INSERT INTO role_permissions (role_id, permission_id) VALUES (?, ?)");
            foreach ($validPerms as $pid) $ins->execute([$role_id,$pid]);
        }
        
        $pdo->commit();
        header('Content-Type:application/json'); 
        echo json_encode(['ok'=>1,'msg'=>'Permisos actualizados']); 
        exit;
    }

    // Save user
    if ($action === 'save_user') {
        $id = isset($_POST['user_id']) ? (int)$_POST['user_id'] : 0;
        $username = trim($_POST['username'] ?? '');
        $full_name = trim($_POST['full_name'] ?? '');
        $password = $_POST['password'] ?? '';

        $rawRoles = $_POST['roles'] ?? [];
        if (!is_array($rawRoles)) $rawRoles = $rawRoles === '' ? [] : [$rawRoles];
        $roleCandidates = array_values(array_filter(array_map(function($v){ return is_numeric($v) ? (int)$v : null; }, $rawRoles), function($v){ return $v !== null && $v > 0; }));

        $active = isset($_POST['active']) ? 1 : 0;

        if ($username === '') { 
            flash('El usuario (username) es requerido.','error'); 
            header('Location:' . site('admin/users.php')); 
            exit; 
        }

        // uniqueness
        if ($id === 0) {
            $st = $pdo->prepare("SELECT COUNT(*) FROM users WHERE username = ?");
            $st->execute([$username]);
            if ((int)$st->fetchColumn() > 0) { 
                flash('El nombre de usuario ya existe.','error'); 
                header('Location:' . site('admin/users.php')); 
                exit; 
            }
        } else {
            $st = $pdo->prepare("SELECT COUNT(*) FROM users WHERE username = ? AND id != ?");
            $st->execute([$username,$id]);
            if ((int)$st->fetchColumn() > 0) { 
                flash('El nombre de usuario ya existe.','error'); 
                header('Location:' . site('admin/users.php')); 
                exit; 
            }
        }

        $pdo->beginTransaction();
        
        if ($id > 0) {
            $parts = []; $params = [];
            $parts[] = "username = ?"; $params[] = $username;
            if ($fullNameCol) { $parts[] = "{$fullNameCol} = ?"; $params[] = $full_name; }
            if ($pwdCol && $password !== '') { $parts[] = "{$pwdCol} = ?"; $params[] = password_hash($password, PASSWORD_DEFAULT); }
            if ($activeCol) { $parts[] = "{$activeCol} = ?"; $params[] = $active; }
            if (!empty($parts)) {
                $params[] = $id;
                $sql = "UPDATE users SET " . implode(', ', $parts) . " WHERE id = ?";
                $pdo->prepare($sql)->execute($params);
            }
            $newUserId = $id;
            flash('Usuario actualizado.');
        } else {
            $cols = ['username']; $place = ['?']; $params = [$username];
            if ($fullNameCol) { $cols[] = $fullNameCol; $place[] = '?'; $params[] = $full_name; }
            if ($pwdCol) { $cols[] = $pwdCol; $place[] = '?'; $params[] = password_hash($password ?: bin2hex(random_bytes(4)), PASSWORD_DEFAULT); }
            if ($activeCol) { $cols[] = $activeCol; $place[] = '?'; $params[] = $active; }
            $sql = "INSERT INTO users (" . implode(',', $cols) . ") VALUES (" . implode(',', $place) . ")";
            $pdo->prepare($sql)->execute($params);
            $newUserId = $pdo->lastInsertId();
            flash('Usuario creado.');
        }

        $validRoleIds = [];
        if (!empty($roleCandidates)) {
            $ph = implode(',', array_fill(0, count($roleCandidates), '?'));
            $st = $pdo->prepare("SELECT id FROM roles WHERE id IN ($ph)");
            $st->execute($roleCandidates);
            $validRoleIds = array_map('intval', $st->fetchAll(PDO::FETCH_COLUMN,0));
            $validRoleIds = array_values(array_unique($validRoleIds));
        }

        $pdo->prepare("DELETE FROM user_roles WHERE user_id = ?")->execute([$newUserId]);
        if (!empty($validRoleIds)) {
            $ins = $pdo->prepare("INSERT INTO user_roles (user_id, role_id) VALUES (?, ?)");
            foreach ($validRoleIds as $rid) $ins->execute([$newUserId, $rid]);
        }

        if ($roleCol) {
            $st = $pdo->prepare("SELECT r.name FROM roles r JOIN user_roles ur ON ur.role_id = r.id WHERE ur.user_id = ? ORDER BY r.id");
            $st->execute([$newUserId]);
            $names = $st->fetchAll(PDO::FETCH_COLUMN, 0);
            if (!empty($names)) {
                $roleText = implode(',', $names);
                $upd = $pdo->prepare("UPDATE users SET {$roleCol} = ? WHERE id = ?");
                $upd->execute([$roleText, $newUserId]);
            } else {
                $upd = $pdo->prepare("UPDATE users SET {$roleCol} = NULL WHERE id = ?");
                $upd->execute([$newUserId]);
            }
        }

        $pdo->commit();
        header('Location:' . site('admin/users.php')); 
        exit;
    }

    // delete user
    if ($action === 'delete_user') {
        $id = isset($_POST['user_id']) ? (int)$_POST['user_id'] : 0;
        if ($id > 0) {
            $pdo->beginTransaction();
            $pdo->prepare("DELETE FROM user_roles WHERE user_id = ?")->execute([$id]);
            $pdo->prepare("DELETE FROM users WHERE id = ?")->execute([$id]);
            $pdo->commit();
            flash('Usuario eliminado.');
        } else flash('Usuario inválido.','error');
        header('Location:' . site('admin/users.php')); 
        exit;
    }

    // fetch_user endpoint
    if (isset($_GET['action']) && $_GET['action'] === 'fetch_user' && isset($_GET['user_id'])) {
        $uid = (int)$_GET['user_id'];
        $select = ["id","username"];
        $select[] = $fullNameCol ? "{$fullNameCol} AS full_name" : "'' AS full_name";
        $select[] = $activeCol ? "{$activeCol} AS active" : "1 AS active";
        $st = $pdo->prepare("SELECT " . implode(', ', $select) . " FROM users WHERE id = ?");
        $st->execute([$uid]);
        $u = $st->fetch(PDO::FETCH_ASSOC) ?: null;
        $rids = [];
        $st2 = $pdo->prepare("SELECT role_id FROM user_roles WHERE user_id = ?");
        $st2->execute([$uid]);
        while ($row = $st2->fetch(PDO::FETCH_ASSOC)) $rids[] = (int)$row['role_id'];
        header('Content-Type: application/json');
        echo json_encode(['ok'=>1,'user'=>$u,'roles'=>$rids]);
        exit;
    }

} catch (Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    error_log("admin/users.php error: " . $e->getMessage());
    flash('Error al procesar la solicitud: ' . $e->getMessage(), 'error');
    header('Location:' . site('admin/users.php')); 
    exit;
}

/* ------------- Render data ------------- */
try {
    $roles = $pdo->query("SELECT * FROM roles ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
    $permissions = $pdo->query("SELECT * FROM permissions ORDER BY id")->fetchAll(PDO::FETCH_ASSOC);

    $selectParts = ["id","username"];
    $selectParts[] = $fullNameCol ? "{$fullNameCol} AS full_name" : "'' AS full_name";
    $selectParts[] = $activeCol ? "{$activeCol} AS active" : "1 AS active";
    $usersSql = "SELECT " . implode(', ', $selectParts) . " FROM users ORDER BY username";
    $users = $pdo->query($usersSql)->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    error_log("admin/users.php (render) SQL error: " . $e->getMessage());
    $users = []; $roles = []; $permissions = [];
    flash('Error al cargar datos: revise error_log.', 'error');
}

$rolePermStmt = $pdo->query("SELECT role_id, permission_id FROM role_permissions");
$rolePerms = []; 
while ($r = $rolePermStmt->fetch(PDO::FETCH_ASSOC)) $rolePerms[$r['role_id']][] = $r['permission_id'];

$userRolesStmt = $pdo->query("SELECT user_id, role_id FROM user_roles");
$userRoles = []; 
while ($r = $userRolesStmt->fetch(PDO::FETCH_ASSOC)) $userRoles[$r['user_id']][] = $r['role_id'];

// ✅ OBTENER PERMISOS DE ARCHIVOS POR ROL (con manejo de errores)
$roleFilePerms = [];
try {
    $roleFilePermsStmt = $pdo->query("SELECT role_id, file_path FROM role_file_permissions");
    while ($r = $roleFilePermsStmt->fetch(PDO::FETCH_ASSOC)) {
        $roleFilePerms[$r['role_id']][] = $r['file_path'];
    }
} catch (Exception $e) {
    error_log("Error cargando role_file_permissions: " . $e->getMessage());
}

$phpFiles = get_php_files();

$flashOk = $_SESSION['flash_ok'] ?? null; 
$flashError = $_SESSION['flash_error'] ?? null;
unset($_SESSION['flash_ok'], $_SESSION['flash_error']);
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Usuarios - Admin</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
  <style>
    body {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      min-height: 100vh;
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }
    
    .main-container {
      background: white;
      border-radius: 20px;
      margin: 30px auto;
      padding: 30px;
      box-shadow: 0 20px 60px rgba(0,0,0,0.3);
    }
    
    .page-header {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      color: white;
      padding: 25px;
      border-radius: 15px;
      margin-bottom: 30px;
    }
    
    .page-header h3 {
      margin: 0;
      font-weight: 700;
    }
    
    .navbar {
      background: white !important;
      box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    }
    
    .card {
      border: none;
      border-radius: 15px;
      box-shadow: 0 5px 15px rgba(0,0,0,0.08);
      transition: all 0.3s ease;
    }
    
    .card:hover {
      box-shadow: 0 10px 30px rgba(0,0,0,0.15);
    }
    
    .card-header {
      background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
      border-radius: 15px 15px 0 0 !important;
      font-weight: 700;
      padding: 15px 20px;
    }
    
    .user-avatar {
      width: 45px;
      height: 45px;
      border-radius: 50%;
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      display: inline-flex;
      align-items: center;
      justify-content: center;
      font-weight: 800;
      color: white;
      font-size: 18px;
    }
    
    .role-item {
      cursor: pointer;
      transition: all 0.3s ease;
      border-radius: 10px;
      margin-bottom: 10px;
      border: 2px solid transparent;
    }
    
    .role-item:hover {
      background: #f8f9fa;
      transform: translateX(5px);
    }
    
    .role-item.role-selected {
      border-color: #667eea;
      background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%);
      box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
    }
    
    .list-group-item {
      border-radius: 10px !important;
      margin-bottom: 10px;
      border: 1px solid #e9ecef;
      transition: all 0.3s ease;
    }
    
    .list-group-item:hover {
      background: #f8f9fa;
      transform: scale(1.02);
    }
    
    .btn {
      border-radius: 10px;
      font-weight: 600;
      transition: all 0.3s ease;
    }
    
    .btn:hover {
      transform: translateY(-2px);
    }
    
    .btn-primary {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      border: none;
    }
    
    .btn-primary:hover {
      box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    }
    
    .modal-content {
      border-radius: 20px;
      border: none;
      box-shadow: 0 20px 60px rgba(0,0,0,0.3);
    }
    
    .modal-header {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      color: white;
      border-radius: 20px 20px 0 0;
    }
    
    .modal-header .btn-close {
      filter: brightness(0) invert(1);
    }
    
    .form-check-input:checked {
      background-color: #667eea;
      border-color: #667eea;
    }
    
    .badge-admin {
      background: linear-gradient(135deg, #ff6b6b 0%, #ee5a6f 100%);
      color: white;
      padding: 6px 12px;
      border-radius: 8px;
      font-weight: 700;
    }
    
    .file-permission-group {
      background: #f8f9fa;
      border-radius: 10px;
      padding: 15px;
      margin-bottom: 15px;
    }
    
    .file-permission-group h6 {
      color: #667eea;
      font-weight: 700;
      margin-bottom: 10px;
    }
    
    .permission-scroll {
      max-height: 400px;
      overflow-y: auto;
    }
    
    .permission-scroll::-webkit-scrollbar {
      width: 8px;
    }
    
    .permission-scroll::-webkit-scrollbar-track {
      background: #f1f1f1;
      border-radius: 10px;
    }
    
    .permission-scroll::-webkit-scrollbar-thumb {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      border-radius: 10px;
    }
  </style>
</head>
<body>

<nav class="navbar navbar-light">
  <div class="container-fluid">
    <a class="navbar-brand fw-bold" href="<?= esc(site('admin/index.php')) ?>">
      <i class="bi bi-people-fill"></i> Gestión de Usuarios
    </a>
    <div class="d-flex gap-2">
      <a class="btn btn-outline-primary btn-sm" href="<?= esc(site('admin/index.php')) ?>">
        <i class="bi bi-arrow-left"></i> Volver
      </a>
      <a class="btn btn-outline-secondary btn-sm" href="<?= esc(site('admin_logout.php')) ?>">
        <i class="bi bi-box-arrow-right"></i> Salir
      </a>
    </div>
  </div>
</nav>

<div class="container-fluid">
  <div class="main-container">
    
    <div class="page-header">
      <div class="d-flex justify-content-between align-items-center">
        <div>
          <h3><i class="bi bi-shield-lock-fill"></i> Usuarios, Roles y Permisos</h3>
          <p class="mb-0 mt-2" style="opacity:0.9">Administra usuarios, roles y control de acceso por archivos</p>
        </div>
        <div class="d-flex gap-2">
          <button class="btn btn-light" data-bs-toggle="modal" data-bs-target="#userModal" onclick="openUserModal()">
            <i class="bi bi-person-plus-fill"></i> Crear Usuario
          </button>
          <button class="btn btn-outline-light" data-bs-toggle="modal" data-bs-target="#roleModal" onclick="openRoleModal()">
            <i class="bi bi-shield-plus"></i> Crear Rol
          </button>
        </div>
      </div>
    </div>

    <?php if ($flashOk): ?>
      <div class="alert alert-success alert-dismissible fade show" role="alert">
        <i class="bi bi-check-circle-fill"></i> <?= esc($flashOk) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
      </div>
    <?php endif; ?>
    
    <?php if ($flashError): ?>
      <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <i class="bi bi-exclamation-triangle-fill"></i> <?= esc($flashError) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
      </div>
    <?php endif; ?>

    <div class="row g-4">
      
      <!-- ROLES -->
      <div class="col-lg-4">
        <div class="card">
          <div class="card-header">
            <i class="bi bi-shield-fill"></i> Roles del Sistema
          </div>
          <div class="card-body">
            <div class="list-group" id="rolesList">
              <?php if (empty($roles)): ?>
                <div class="text-muted text-center py-4">
                  <i class="bi bi-inbox" style="font-size:48px;opacity:0.3"></i>
                  <p>No hay roles creados</p>
                </div>
              <?php else: ?>
                <?php foreach ($roles as $r): ?>
                  <div class="list-group-item role-item" data-role-id="<?= $r['id'] ?>" data-is-admin="<?= $r['is_admin'] ?? 0 ?>">
                    <div class="d-flex justify-content-between align-items-start">
                      <div class="flex-grow-1">
                        <div class="d-flex align-items-center gap-2 mb-1">
                          <i class="bi bi-shield-fill-check text-primary"></i>
                          <strong><?= esc($r['name']) ?></strong>
                          <?php if (!empty($r['is_admin'])): ?>
                            <span class="badge-admin">ADMIN</span>
                          <?php endif; ?>
                        </div>
                        <div class="small text-muted"><?= esc($r['description']) ?></div>
                      </div>
                      <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary" onclick="editRole(event, <?= $r['id'] ?>)">
                          <i class="bi bi-pencil-fill"></i>
                        </button>
                        <form method="post" style="display:inline" onsubmit="return confirm('¿Eliminar rol?')">
                          <input type="hidden" name="action" value="delete_role">
                          <input type="hidden" name="role_id" value="<?= $r['id'] ?>">
                          <button class="btn btn-outline-danger">
                            <i class="bi bi-trash-fill"></i>
                          </button>
                        </form>
                      </div>
                    </div>
                  </div>
                <?php endforeach; ?>
              <?php endif; ?>
            </div>
          </div>
        </div>
      </div>

      <!-- PERMISOS POR ARCHIVOS -->
      <div class="col-lg-4">
        <div class="card">
          <div class="card-header">
            <i class="bi bi-file-earmark-lock-fill"></i> Control de Acceso por Archivos
          </div>
          <div class="card-body">
            <div id="selectedRoleInfo" class="alert alert-info">
              <i class="bi bi-info-circle-fill"></i>
              Selecciona un rol para configurar sus permisos
            </div>
            
            <div id="filePermEditor" style="display:none">
              <div class="alert alert-warning" id="adminWarning" style="display:none">
                <i class="bi bi-shield-fill-exclamation"></i>
                <strong>Rol Administrador:</strong> Tiene acceso total automático a todos los archivos.
              </div>
              
              <form id="filePermForm" onsubmit="return false;">
                <input type="hidden" id="file_perm_role_id" value="">
                
                <div class="permission-scroll">
                  <?php 
                  $grouped = [];
                  foreach ($phpFiles as $file) {
                    $grouped[$file['group']][] = $file;
                  }
                  
                  foreach ($grouped as $group => $groupFiles): 
                  ?>
                    <div class="file-permission-group">
                      <h6><i class="bi bi-folder-fill"></i> <?= esc($group) ?></h6>
                      <?php foreach ($groupFiles as $file): ?>
                        <div class="form-check">
                          <input class="form-check-input file-checkbox" type="checkbox" value="<?= esc($file['file']) ?>" id="file_<?= esc(md5($file['file'])) ?>">
                          <label class="form-check-label" for="file_<?= esc(md5($file['file'])) ?>">
                            <i class="bi bi-file-earmark-code"></i> <?= esc(basename($file['file'])) ?>
                          </label>
                        </div>
                      <?php endforeach; ?>
                    </div>
                  <?php endforeach; ?>
                </div>
                
                <div class="d-flex gap-2 mt-3">
                  <button type="button" class="btn btn-primary" id="saveFilePermsBtn">
                    <i class="bi bi-save-fill"></i> Guardar Permisos
                  </button>
                  <button type="button" class="btn btn-secondary" id="cancelFilePermsBtn">
                    <i class="bi bi-x-circle"></i> Cancelar
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      </div>

      <!-- USUARIOS -->
      <div class="col-lg-4">
        <div class="card">
          <div class="card-header">
            <i class="bi bi-people-fill"></i> Usuarios del Sistema
          </div>
          <div class="card-body">
            <div class="list-group">
              <?php if (empty($users)): ?>
                <div class="text-muted text-center py-4">
                  <i class="bi bi-inbox" style="font-size:48px;opacity:0.3"></i>
                  <p>No hay usuarios registrados</p>
                </div>
              <?php else: ?>
                <?php foreach ($users as $u): ?>
                  <div class="list-group-item">
                    <div class="d-flex gap-3 align-items-center">
                      <div class="user-avatar"><?= esc(strtoupper(substr($u['username'],0,1))) ?></div>
                      <div class="flex-grow-1">
                        <div class="d-flex align-items-center gap-2 mb-1">
                          <strong><?= esc($u['username']) ?></strong>
                          <?php if (!(int)($u['active'] ?? 1)): ?>
                            <span class="badge bg-danger">Inactivo</span>
                          <?php endif; ?>
                        </div>
                        <div class="small text-muted"><?= esc($u['full_name'] ?? '') ?></div>
                        <div class="small">
                          <i class="bi bi-shield-fill-check"></i>
                          <?= esc(implode(', ', array_map(function($rid) use ($pdo){ 
                            $st = $pdo->prepare("SELECT name FROM roles WHERE id = ?"); 
                            $st->execute([$rid]); 
                            return $st->fetchColumn(); 
                          }, $userRoles[$u['id']] ?? [])) ?: 'Sin rol') ?>
                        </div>
                      </div>
                      <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary" onclick="openUserModal(<?= $u['id'] ?>)">
                          <i class="bi bi-pencil-fill"></i>
                        </button>
                        <form method="post" style="display:inline" onsubmit="return confirm('¿Eliminar usuario?')">
                          <input type="hidden" name="action" value="delete_user">
                          <input type="hidden" name="user_id" value="<?= $u['id'] ?>">
                          <button class="btn btn-outline-danger">
                            <i class="bi bi-trash-fill"></i>
                          </button>
                        </form>
                      </div>
                    </div>
                  </div>
                <?php endforeach; ?>
              <?php endif; ?>
            </div>
          </div>
        </div>
      </div>

    </div>
  </div>
</div>

<!-- MODAL: CREAR/EDITAR ROL -->
<div class="modal fade" id="roleModal" tabindex="-1">
  <div class="modal-dialog">
    <form class="modal-content" method="post" id="roleForm">
      <input type="hidden" name="action" value="save_role">
      <input type="hidden" name="role_id" id="role_id" value="">
      <div class="modal-header">
        <h5 class="modal-title" id="roleModalTitle">
          <i class="bi bi-shield-plus"></i> Crear Rol
        </h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <div class="mb-3">
          <label class="form-label fw-bold">
            <i class="bi bi-tag-fill"></i> Nombre del Rol
          </label>
          <input class="form-control form-control-lg" name="name" id="role_name" required placeholder="Ej: Cajero, Supervisor">
        </div>
        <div class="mb-3">
          <label class="form-label fw-bold">
            <i class="bi bi-card-text"></i> Descripción
          </label>
          <textarea class="form-control" name="description" id="role_desc" rows="3" placeholder="Describe las responsabilidades del rol"></textarea>
        </div>
        <div class="mb-3">
          <div class="form-check form-switch">
            <input class="form-check-input" type="checkbox" id="is_admin" name="is_admin">
            <label class="form-check-label fw-bold" for="is_admin">
              <i class="bi bi-shield-fill-exclamation"></i> Es Administrador
              <small class="text-muted d-block">Los administradores tienen acceso total sin restricciones</small>
            </label>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-secondary" type="button" data-bs-dismiss="modal">
          <i class="bi bi-x-circle"></i> Cancelar
        </button>
        <button class="btn btn-primary" type="submit">
          <i class="bi bi-save-fill"></i> Guardar Rol
        </button>
      </div>
    </form>
  </div>
</div>

<!-- MODAL: CREAR/EDITAR USUARIO -->
<div class="modal fade" id="userModal" tabindex="-1">
  <div class="modal-dialog modal-lg">
    <form class="modal-content" method="post" id="userForm">
      <input type="hidden" name="action" value="save_user">
      <input type="hidden" name="user_id" id="user_id" value="">
      <div class="modal-header">
        <h5 class="modal-title" id="userModalTitle">
          <i class="bi bi-person-plus-fill"></i> Crear Usuario
        </h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <div class="row g-3">
          <div class="col-md-6">
            <label class="form-label fw-bold">
              <i class="bi bi-person-fill"></i> Usuario (Login)
            </label>
            <input class="form-control form-control-lg" name="username" id="username" required>
          </div>
          <div class="col-md-6">
            <label class="form-label fw-bold">
              <i class="bi bi-id-card"></i> Nombre Completo
            </label>
            <input class="form-control form-control-lg" name="full_name" id="full_name">
          </div>
          <div class="col-md-6">
            <label class="form-label fw-bold">
              <i class="bi bi-key-fill"></i> Contraseña
            </label>
            <input type="password" class="form-control form-control-lg" name="password" id="password" placeholder="Dejar vacío para no cambiar">
          </div>
          <div class="col-md-6">
            <label class="form-label fw-bold">
              <i class="bi bi-shield-check"></i> Roles
            </label>
            <select class="form-select form-select-lg" id="user_roles" name="roles[]" multiple required>
              <?php foreach ($roles as $r): ?>
                <option value="<?= $r['id'] ?>"><?= esc($r['name']) ?></option>
              <?php endforeach; ?>
            </select>
            <small class="text-muted">Mantén presionada Ctrl/Cmd para seleccionar varios</small>
          </div>
          <div class="col-12">
            <div class="form-check form-switch">
              <input class="form-check-input" type="checkbox" id="user_active" name="active" checked>
              <label class="form-check-label fw-bold" for="user_active">
                <i class="bi bi-check-circle-fill"></i> Usuario Activo
              </label>
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-secondary" type="button" data-bs-dismiss="modal">
          <i class="bi bi-x-circle"></i> Cancelar
        </button>
        <button class="btn btn-primary" type="submit">
          <i class="bi bi-save-fill"></i> Guardar Usuario
        </button>
      </div>
    </form>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
const roleFilePerms = <?= json_encode($roleFilePerms ?: []) ?>;
const rolePerms = <?= json_encode($rolePerms ?: []) ?>;
const roles = <?= json_encode($roles ?: []) ?>;

// Seleccionar rol y mostrar editor de permisos de archivos
document.querySelectorAll('.role-item').forEach(el => {
  el.addEventListener('click', function(e) {
    if (e.target.tagName === 'BUTTON' || e.target.closest('form')) return;
    
    const rid = parseInt(this.dataset.roleId);
    const isAdmin = parseInt(this.dataset.isAdmin || 0);
    
    selectRole(rid, isAdmin);
  });
});

function selectRole(roleId, isAdmin = 0) {
  document.querySelectorAll('.role-item').forEach(el => 
    el.classList.toggle('role-selected', el.dataset.roleId == roleId)
  );
  
  document.getElementById('filePermEditor').style.display = 'block';
  document.getElementById('selectedRoleInfo').style.display = 'none';
  document.getElementById('file_perm_role_id').value = roleId;
  
  const role = roles.find(r => r.id == roleId);
  const roleName = role ? role.name : 'ID ' + roleId;
  
  // Mostrar advertencia si es admin
  if (isAdmin) {
    document.getElementById('adminWarning').style.display = 'block';
    document.querySelectorAll('.file-checkbox').forEach(cb => {
      cb.checked = true;
      cb.disabled = true;
    });
  } else {
    document.getElementById('adminWarning').style.display = 'none';
    // Limpiar y cargar permisos
    document.querySelectorAll('.file-checkbox').forEach(cb => {
      cb.checked = false;
      cb.disabled = false;
    });
    
    const files = roleFilePerms[roleId] || [];
    files.forEach(file => {
      const el = document.getElementById('file_' + md5(file));
      if (el) el.checked = true;
    });
  }
}

function md5(str) {
  let hash = 0;
  for (let i = 0; i < str.length; i++) {
    hash = ((hash << 5) - hash) + str.charCodeAt(i);
    hash = hash & hash;
  }
  return Math.abs(hash).toString(16);
}

// Guardar permisos de archivos
document.getElementById('saveFilePermsBtn').addEventListener('click', async function() {
  const roleId = parseInt(document.getElementById('file_perm_role_id').value || 0);
  if (!roleId) return alert('Seleccione un rol.');
  
  const checked = Array.from(document.querySelectorAll('.file-checkbox:checked:not(:disabled)'))
    .map(cb => cb.value);
  
  try {
    const res = await fetch('<?= esc(site('admin/users.php')) ?>?action=save_file_permissions', {
      method: 'POST',
      headers: {'Content-Type':'application/json'},
      body: JSON.stringify({ role_id: roleId, files: checked })
    });
    const j = await res.json();
    if (j.ok) { 
      alert('✅ Permisos de archivos guardados correctamente'); 
      location.reload(); 
    } else {
      alert('❌ Error: ' + (j.msg || 'Desconocido'));
    }
  } catch(e) { 
    console.error(e); 
    alert('❌ Error guardando permisos'); 
  }
});

document.getElementById('cancelFilePermsBtn').addEventListener('click', function() {
  document.getElementById('filePermEditor').style.display = 'none';
  document.getElementById('selectedRoleInfo').style.display = 'block';
  document.querySelectorAll('.role-item').forEach(el => el.classList.remove('role-selected'));
});

// Editar rol
function editRole(e, id) {
  e.stopPropagation();
  e.preventDefault();
  const role = roles.find(r => r.id == id);
  if (!role) return alert('Rol no encontrado');
  
  document.getElementById('role_id').value = id;
  document.getElementById('role_name').value = role.name || '';
  document.getElementById('role_desc').value = role.description || '';
  document.getElementById('is_admin').checked = role.is_admin ? true : false;
  document.getElementById('roleModalTitle').innerHTML = '<i class="bi bi-pencil-fill"></i> Editar Rol';
  
  new bootstrap.Modal(document.getElementById('roleModal')).show();
}

function openRoleModal() {
  document.getElementById('roleForm').reset();
  document.getElementById('role_id').value = '';
  document.getElementById('roleModalTitle').innerHTML = '<i class="bi bi-shield-plus"></i> Crear Rol';
}

// Abrir modal de usuario
async function openUserModal(userId = 0) {
  document.getElementById('userForm').reset();
  document.getElementById('user_id').value = '';
  Array.from(document.getElementById('user_roles').options).forEach(o => o.selected = false);
  document.getElementById('userModalTitle').innerHTML = userId ? '<i class="bi bi-pencil-fill"></i> Editar Usuario' : '<i class="bi bi-person-plus-fill"></i> Crear Usuario';
  
  if (userId) {
    try {
      const res = await fetch('<?= esc(site('admin/users.php')) ?>?action=fetch_user&user_id=' + userId);
      const j = await res.json();
      if (j.ok && j.user) {
        document.getElementById('user_id').value = j.user.id;
        document.getElementById('username').value = j.user.username;
        document.getElementById('full_name').value = j.user.full_name || '';
        document.getElementById('password').value = '';
        document.getElementById('user_active').checked = j.user.active ? true : false;
        
        if (Array.isArray(j.roles)) {
          j.roles.forEach(rid => {
            const opt = document.querySelector('#user_roles option[value="'+rid+'"]');
            if (opt) opt.selected = true;
          });
        }
      } else {
        alert('No se pudo cargar la información del usuario.');
      }
    } catch(e) { 
      console.error(e); 
      alert('Error cargando usuario'); 
    }
  }
  
  new bootstrap.Modal(document.getElementById('userModal')).show();
}
</script>

</body>
</html>