<?php
// Incluir configuración desde el archivo db.php
$config = include('db.php');

try {
    $dsn = "mysql:host={$config['host']};port={$config['port']};dbname={$config['dbname']};charset={$config['charset']}";
    $pdo = new PDO($dsn, $config['user'], $config['pass']);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("Error al conectar a la base de datos: " . $e->getMessage());
}

// Inicializar variables de sesión y generar un session_id único para identificar cada operación
session_start();

/*
  OBJETIVO:
  - Un "cuadre" = 1 session_id.
  - Mientras registres varios renglones (efectivo/tarjeta/etc) del MISMO cuadre => mismo session_id.
  - Cuando terminas y le das "Imprimir y Limpiar" => el cuadre se cierra y el PROXIMO registro debe iniciar con OTRO session_id nuevo.
*/

// Si NO hay session_id, crear uno (primer cuadre)
if (!isset($_SESSION['session_id']) || empty($_SESSION['session_id'])) {
    $_SESSION['session_id'] = uniqid('', true);
}
$session_id = $_SESSION['session_id'];

$nombreBombero = $_POST['nombre_bombero'] ?? '';
$fechaInicio = $_POST['fecha_inicio'] ?? '';
$fechaFinal = $_POST['fecha_final'] ?? '';
$horaInicio = $_POST['hora_inicio'] ?? '';
$horaFinal = $_POST['hora_final'] ?? '';
$bomba1 = $_POST['bomba1'] ?? '';
$bomba2 = $_POST['bomba2'] ?? '';
$tipodecobro = $_POST['tipodecobro'] ?? '';
$mensaje = "";

// Obtener bombas disponibles
$stmt = $pdo->query("SELECT * FROM pumps");
$pumps = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Procesar registro
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['action']) && $_POST['action'] === 'register') {
    $valor = $_POST['valor'];

    $sql = "INSERT INTO cuadre_bomberos 
            (nombre_bombero, fecha_inicio, fecha_final, hora_inicio, hora_final, bomba1, bomba2, tipodecobro, valor, session_id) 
            VALUES (:nombre_bombero, :fecha_inicio, :fecha_final, :hora_inicio, :hora_final, :bomba1, :bomba2, :tipodecobro, :valor, :session_id)";
    $stmt = $pdo->prepare($sql);

    $params = [
        ':nombre_bombero' => $nombreBombero,
        ':fecha_inicio' => $fechaInicio,
        ':fecha_final' => $fechaFinal,
        ':hora_inicio' => $horaInicio,
        ':hora_final' => $horaFinal,
        ':bomba1' => $bomba1,
        ':bomba2' => $bomba2,
        ':tipodecobro' => $tipodecobro,
        ':valor' => $valor,
        ':session_id' => $session_id
    ];

    $stmt->execute($params);

    $sqlHistorico = "INSERT INTO cuadre_bomberos_historico 
                     (nombre_bombero, fecha_inicio, fecha_final, hora_inicio, hora_final, bomba1, bomba2, tipodecobro, valor, session_id) 
                     VALUES (:nombre_bombero, :fecha_inicio, :fecha_final, :hora_inicio, :hora_final, :bomba1, :bomba2, :tipodecobro, :valor, :session_id)";
    $stmtHistory = $pdo->prepare($sqlHistorico);
    $stmtHistory->execute($params);

    $mensaje = "Cuadre registrado exitosamente.";
}

// Procesar edición
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['action']) && $_POST['action'] === 'edit') {
    $idToEdit = $_POST['id'];
    $newValor = $_POST['valor_edit'];
    $stmtEdit = $pdo->prepare("UPDATE cuadre_bomberos SET valor = :valor WHERE id = :id");
    $stmtEdit->execute([
        ':valor' => $newValor,
        ':id' => $idToEdit,
    ]);
    $mensaje = "Registro editado correctamente.";
}

// Procesar eliminación
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['action']) && $_POST['action'] === 'delete') {
    $idToDelete = $_POST['id'];
    $pdo->prepare("DELETE FROM cuadre_bomberos WHERE id = :id")->execute([':id' => $idToDelete]);
    $mensaje = "Registro eliminado correctamente.";
}

// Procesar "Generar reporte"
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['action']) && $_POST['action'] === 'print_report') {
    // Guardar el session_id actual ANTES de cambiarlo, para imprimir el cuadre correcto
    $session_id_to_print = $session_id;

    // Limpiar tabla temporal SOLO de ese cuadre
    $stmtDelete = $pdo->prepare("DELETE FROM cuadre_bomberos WHERE session_id = :session_id");
    $stmtDelete->execute([':session_id' => $session_id_to_print]);

    // AQUI ESTA LA CORRECCION CLAVE:
    // Generar un NUEVO session_id para el siguiente cuadre (en la misma sesión PHP)
    $_SESSION['session_id'] = uniqid('', true);
    $session_id = $_SESSION['session_id'];

    // Enviar el session_id del cuadre que se va a imprimir
    header("Location: generar-reporte.php?session_id=" . urlencode($session_id_to_print));
    exit();
}

// Obtener los registros actuales (solo los de la sesión actual)
$registrosQuery = $pdo->prepare("SELECT * FROM cuadre_bomberos WHERE session_id = :session_id ORDER BY id DESC");
$registrosQuery->execute([':session_id' => $session_id]);
$registros = $registrosQuery->fetchAll(PDO::FETCH_ASSOC);

// Calcular total de los valores registrados de la sesión actual
$stmtTotal = $pdo->prepare("SELECT SUM(valor) AS total FROM cuadre_bomberos WHERE session_id = :session_id");
$stmtTotal->execute([':session_id' => $session_id]);
$totalValor = $stmtTotal->fetchColumn();
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cuadre de Bomberos</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css">
</head>
<body>
<header class="bg-primary text-white p-3">
    <div class="container d-flex justify-content-between align-items-center">
        <h2 class="m-0">Fuel POS - Cuadre de Bomberos</h2>
        <div>
            <a href="index.php" class="btn btn-light btn-sm">Inicio</a>
            <a href="historico-cuadres.php" class="btn btn-light btn-sm">Histórico de Cuadres</a>
        </div>
    </div>
</header>
<main class="container py-4">
    <div class="card shadow">
        <div class="card-body">
            <h1 class="card-title mb-4 text-center">Registrar Cuadre de Bomberos</h1>
            <?php if (!empty($mensaje)): ?>
                <div class="alert alert-success"><?= $mensaje ?></div>
            <?php endif; ?>
            <form method="POST" class="row g-3">
                <input type="hidden" name="action" value="register">
                <div class="col-md-6">
                    <label class="form-label">Nombre del Bombero:</label>
                    <input type="text" name="nombre_bombero" class="form-control" value="<?= htmlspecialchars($nombreBombero) ?>" required>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Fecha Inicio:</label>
                    <input type="date" name="fecha_inicio" class="form-control" value="<?= htmlspecialchars($fechaInicio) ?>" required>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Fecha Final:</label>
                    <input type="date" name="fecha_final" class="form-control" value="<?= htmlspecialchars($fechaFinal) ?>" required>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Hora Inicio:</label>
                    <input type="time" name="hora_inicio" class="form-control" value="<?= htmlspecialchars($horaInicio) ?>" required>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Hora Final:</label>
                    <input type="time" name="hora_final" class="form-control" value="<?= htmlspecialchars($horaFinal) ?>" required>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Bomba 1:</label>
                    <select name="bomba1" class="form-select" required>
                        <?php foreach ($pumps as $pump): ?>
                            <option value="<?= $pump['id'] ?>" <?= $pump['id'] == $bomba1 ? 'selected' : '' ?>>
                                <?= htmlspecialchars($pump['name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Bomba 2:</label>
                    <select name="bomba2" class="form-select" required>
                        <?php foreach ($pumps as $pump): ?>
                            <option value="<?= $pump['id'] ?>" <?= $pump['id'] == $bomba2 ? 'selected' : '' ?>>
                                <?= htmlspecialchars($pump['name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Tipo de Cobro:</label>
                    <select name="tipodecobro" class="form-select" required>
                        <option value="efectivo" <?= $tipodecobro === 'efectivo' ? 'selected' : '' ?>>Efectivo</option>
                        <option value="tarjeta" <?= $tipodecobro === 'tarjeta' ? 'selected' : '' ?>>Tarjeta</option>
                        <option value="credito" <?= $tipodecobro === 'credito' ? 'selected' : '' ?>>Crédito</option>
                        <option value="otros" <?= $tipodecobro === 'otros' ? 'selected' : '' ?>>Otros</option>
                    </select>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Valor:</label>
                    <input type="number" name="valor" class="form-control" step="0.01" required>
                </div>
                <div class="col-12 text-center">
                    <button type="submit" class="btn btn-primary">Registrar</button>
                </div>
            </form>

            <?php if (!empty($registros)): ?>
            <h2 class="mt-5">Registros Actuales</h2>
            <table class="table table-striped table-bordered mt-4">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Bombero</th>
                        <th>Fecha Inicio</th>
                        <th>Fecha Final</th>
                        <th>Hora Inicio</th>
                        <th>Hora Final</th>
                        <th>Tipo de Cobro</th>
                        <th>Valor</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($registros as $registro): ?>
                    <tr>
                        <td><?= htmlspecialchars($registro['id']) ?></td>
                        <td><?= htmlspecialchars($registro['nombre_bombero']) ?></td>
                        <td><?= htmlspecialchars($registro['fecha_inicio']) ?></td>
                        <td><?= htmlspecialchars($registro['fecha_final']) ?></td>
                        <td><?= htmlspecialchars($registro['hora_inicio']) ?></td>
                        <td><?= htmlspecialchars($registro['hora_final']) ?></td>
                        <td><?= htmlspecialchars($registro['tipodecobro']) ?></td>
                        <td><?= htmlspecialchars(number_format($registro['valor'], 2)) ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <div class="text-center">
                <form method="POST">
                    <input type="hidden" name="action" value="print_report">
                    <button type="submit" class="btn btn-success">Imprimir y Limpiar</button>
                </form>
            </div>
            <?php endif; ?>
        </div>
    </div>
</main>
</body>
</html>