<?php
/**
 * listen_events.php
 * Escucha eventos desde Fusion/Spirit por PuertoEvento (default 3011) como CLIENTE TCP/UDP.
 * Modo CLI recomendado.
 *
 * CLI ejemplos:
 *   php listen_events.php --host=172.17.125.102 --port=3011 --proto=tcp
 *   php listen_events.php --host=172.17.125.102 --port=3011 --proto=udp
 *
 * Si en tu instalación los eventos llegan por conexión entrante (server->cliente),
 * entonces este script no aplica; necesitarías abrir un socket servidor local
 * y configurar Fusion para "push" hacia tu IP/puerto. Aquí implementamos cliente.
 */
declare(strict_types=1);

function is_cli(): bool {
    return php_sapi_name() === 'cli' || PHP_SAPI === 'cli';
}

function connect_stream(string $host, int $port, string $proto, float $timeout=5.0) {
    $scheme = strtolower($proto).'://'.$host.':'.$port;
    $ctx = stream_context_create();
    $fp = @stream_socket_client($scheme, $errno, $errstr, $timeout, STREAM_CLIENT_CONNECT, $ctx);
    if (!$fp) {
        fwrite(STDERR, "Conexión fallida: $errstr ($errno)\n");
        return null;
    }
    stream_set_timeout($fp, 0);
    return $fp;
}

if (!is_cli()) {
    header('Content-Type: text/plain; charset=utf-8');
    echo "Uso solo por CLI.\n";
    exit;
}

$args = getopt('', ['host:', 'port::', 'proto::', 'timeout::']);
$host = $args['host'] ?? '127.0.0.1';
$port = isset($args['port']) ? (int)$args['port'] : 3011;
$proto = $args['proto'] ?? 'tcp';
$timeout = isset($args['timeout']) ? (float)$args['timeout'] : 5.0;

$fp = connect_stream($host,$port,$proto,$timeout);
if (!$fp) exit(2);

echo "Conectado a $proto://$host:$port\nEscuchando eventos...\n(Ctrl+C para salir)\n\n";

$buf = '';
while (true) {
    $r = [$fp]; $w = null; $e = null;
    $n = @stream_select($r, $w, $e, 1);
    if ($n === false) { usleep(200000); continue; }
    if ($n > 0) {
        $chunk = @fread($fp, 8192);
        if ($chunk === '' || $chunk === false) { usleep(50000); continue; }
        $buf .= $chunk;
        // imprimir por líneas cuando haya \n
        while (($pos = strpos($buf, "\n")) !== false) {
            $line = substr($buf, 0, $pos);
            $buf = substr($buf, $pos+1);
            echo "[".date('H:i:s')."] ".$line."\n";
        }
    }
}