<?php
// Incluir configuración desde el archivo db.php
$config = include('db.php');
try {
    $dsn = "mysql:host={$config['host']};port={$config['port']};dbname={$config['dbname']};charset=utf8";
    $pdo = new PDO($dsn, $config['user'], $config['pass']);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("Error al conectar a la base de datos: " . $e->getMessage());
}

// Usar session_id para filtrar los registros más recientes
session_start();

/*
  CORRECCION:
  - cuadrebomberos.php ahora redirige a:
      generar-reporte.php?session_id=XXXX
  - Por eso aquí debemos tomar primero el session_id por GET.
  - Si no viene por GET, usamos el de la sesión como respaldo.
*/
$session_id = $_GET['session_id'] ?? ($_SESSION['session_id'] ?? '');

// Si no existe session_id válido, detener con mensaje claro
if (empty($session_id)) {
    die("Error: No se recibió un session_id válido para generar el reporte.");
}

// Obtener registros de la tabla histórica filtrados por session_id
$reportDataQuery = $pdo->prepare("SELECT * FROM cuadre_bomberos_historico WHERE session_id = :session_id ORDER BY id ASC");
$reportDataQuery->execute([':session_id' => $session_id]);
$reportData = $reportDataQuery->fetchAll(PDO::FETCH_ASSOC);

// Calcular el total de los valores registrados en la operación actual
$stmtTotal = $pdo->prepare("SELECT SUM(valor) AS total FROM cuadre_bomberos_historico WHERE session_id = :session_id");
$stmtTotal->execute([':session_id' => $session_id]);
$totalValor = $stmtTotal->fetchColumn();

// Obtener información de la estación
$stationQuery = $pdo->query("SELECT * FROM stations LIMIT 1");
$stationData = $stationQuery->fetch(PDO::FETCH_ASSOC);

// Establecer el nombre del primer bombero de los registros actuales
$nombreBombero = $reportData[0]['nombre_bombero'] ?? 'Desconocido';

// NO eliminamos la tabla "cuadre_bomberos_historico", ya que contiene el historial completo.
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Reporte de Cuadre</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            text-align: center;
        }
        h1, h2 {
            margin-bottom: 10px;
        }
        .details {
            margin: 20px 0;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
        }
        th {
            background-color: #f3f3f3;
        }
        .signatures {
            display: flex;
            justify-content: space-around;
            margin-top: 40px;
        }
        .signatures div {
            text-align: center;
        }
        .button-container {
            margin-top: 20px;
        }
        .button-container button {
            padding: 10px 20px;
            font-size: 16px;
            border: none;
            border-radius: 5px;
            background-color: #0056b3;
            color: white;
            cursor: pointer;
        }
    </style>
</head>
<body onload="window.print();">
    <div>
        <h1><?= htmlspecialchars($stationData['nombre'] ?? 'Estación Desconocida') ?></h1>
        <p class="details">
            <?= htmlspecialchars($stationData['direccion'] ?? 'Dirección no disponible') ?><br>
            Teléfono: <?= htmlspecialchars($stationData['telefono'] ?? 'No disponible') ?><br>
            RNC: <?= htmlspecialchars($stationData['rnc'] ?? 'No disponible') ?>
        </p>
        <h2>Detalles</h2>
        <p>Bombero: <?= htmlspecialchars($nombreBombero) ?></p>
        <table>
            <thead>
                <tr>
                    <th>Tipo de Cobro</th>
                    <th>Valor</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($reportData as $record): ?>
                <tr>
                    <td><?= htmlspecialchars($record['tipodecobro']) ?></td>
                    <td><?= htmlspecialchars(number_format($record['valor'], 2)) ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <p>Total: <?= htmlspecialchars(number_format($totalValor, 2)) ?></p>
        <div class="signatures">
            <div>
                <p>_______________________</p>
                <p>Firma del Bombero</p>
            </div>
            <div>
                <p>_______________________</p>
                <p>Firma del Supervisor</p>
            </div>
        </div>
        <div class="button-container">
            <button onclick="window.location.href='cuadrebomberos.php';">Volver</button>
        </div>
    </div>
</body>
</html>