<?php
// init.php - bootstrap y utilidades (con timezone configurable)
$config = require __DIR__ . '/config.php';

// Set timezone early so all date()/DateTime() use correct TZ.
// Use config->timezone if provided, otherwise default to America/Santo_Domingo.
$tz = !empty($config->timezone) ? $config->timezone : 'America/Santo_Domingo';
date_default_timezone_set($tz);

// ensure upload_dir exists
if (!empty($config->upload_dir) && !file_exists($config->upload_dir)) {
    mkdir($config->upload_dir, 0777, true);
}

// session
session_name($config->session_name ?? 'fuelpos_session');
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * pdo()
 * Retorna una instancia singleton de PDO usando la configuración en $config.
 */
function pdo() {
    static $pdo;
    global $config;
    if ($pdo) return $pdo;
    $dsn = "mysql:host={$config->db_host};dbname={$config->db_name};charset=utf8mb4";
    $pdo = new PDO($dsn, $config->db_user, $config->db_pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);
    return $pdo;
}

/**
 * Exponer la conexión PDO en variables comunes para compatibilidad
 * con código legado que espera $pdo, $db, $dbh o $conn.
 *
 * Si la creación de PDO falla, guardamos el mensaje en $pdo_init_error
 * y dejamos las variables en null para que las páginas puedan manejarlo.
 */
$pdo = null;
$pdo_init_error = null;
try {
    // intenta inicializar la conexión PDO (no obligatorio que exista inmediatamente)
    $pdo = pdo();
} catch (Throwable $e) {
    // No hacemos fatal: otras partes del sistema pueden manejar la ausencia de DB.
    $pdo = null;
    $pdo_init_error = $e->getMessage();
}

// Aliases para compatibilidad con distintos nombres usados en proyectos
// (si alguna ya existe en init.php antes, no sobrescribimos)
if (!isset($db)) $db = $pdo;
if (!isset($dbh)) $dbh = $pdo;
if (!isset($conn)) $conn = $pdo;

/**
 * site($path='')
 * Devuelve la URL base del proyecto + $path.
 * - Si config->base_url está definido y no es localhost/127.0.0.1, se usa tal cual.
 * - Si no, construye la URL a partir de DOCUMENT_ROOT y la ruta física del proyecto.
 */
function site($path = '') {
    global $config;

    $cfg = trim((string)($config->base_url ?? ''));

    // si hay base_url explícita y no es localhost, la usamos
    if ($cfg !== '' && stripos($cfg, 'localhost') === false && stripos($cfg, '127.0.0.1') === false) {
        $base = rtrim($cfg, '/');
    } else {
        // construimos dinámicamente usando DOCUMENT_ROOT y la ruta física del proyecto
        if (php_sapi_name() === 'cli' || empty($_SERVER['HTTP_HOST'])) {
            $host = 'localhost';
            $scheme = 'http';
            $projectUri = '';
        } else {
            $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
            $host = $_SERVER['HTTP_HOST'];

            $docRoot = realpath($_SERVER['DOCUMENT_ROOT']);
            $projDir = realpath(__DIR__); // carpeta del proyecto (donde está init.php)

            if ($docRoot !== false && $projDir !== false && strpos($projDir, $docRoot) === 0) {
                // ruta relativa del proyecto respecto a DOCUMENT_ROOT
                $rel = substr($projDir, strlen($docRoot));
                $rel = str_replace('\\', '/', $rel);
                $projectUri = rtrim($rel, '/');
            } else {
                // fallback: usar dirname of SCRIPT_NAME (works in many setups)
                $scriptDir = rtrim(str_replace('\\','/', dirname($_SERVER['SCRIPT_NAME'])), '/');
                $projectUri = ($scriptDir === '/' || $scriptDir === '.') ? '' : $scriptDir;
            }
        }
        $base = rtrim($scheme . '://' . $host . $projectUri, '/');
    }

    $base = rtrim($base, '/');
    $path = ltrim((string)$path, '/');
    return $path === '' ? $base . '/' : $base . '/' . $path;
}

/**
 * project_url($path='')
 * URL absoluta apuntando siempre a la raíz del proyecto.
 */
function project_url($path = '') {
    return site($path);
}

function is_logged() {
    return !empty($_SESSION['user_id']);
}

function require_login() {
    if (!is_logged()) {
        header('Location: ' . site('admin_login.php'));
        exit;
    }
}

/**
 * Helpers opcionales que pueden usarse en el proyecto:
 * - get_pdo() retorna el PDO si está disponible, o null.
 * - get_db_error() retorna el mensaje de error si la inicialización falló.
 */
function get_pdo() {
    global $pdo;
    return $pdo instanceof PDO ? $pdo : null;
}
function get_db_error() {
    global $pdo_init_error;
    return $pdo_init_error;
}

/* Fin de init.php */