<?php
/**
 * sync_sales.php
 * Lee C:/opscode/sale.txt y registra ventas nuevas
 * Llamado desde api.php?action=sync_sales_from_file
 */

require_once __DIR__ . '/init.php';

$pdo = pdo();
$salesFile = 'C:/opscode/sale.txt';
$processedFile = __DIR__ . '/data/sales_processed.json';
$logFile = __DIR__ . '/logs/sync_sales.log';

// Crear carpetas si no existen
@mkdir(__DIR__ . '/data', 0755, true);
@mkdir(__DIR__ . '/logs', 0755, true);

// Verificar archivo
if (!file_exists($salesFile)) {
    echo json_encode(['ok'=>0,'error'=>'Archivo sale.txt no encontrado','path'=>$salesFile]);
    exit;
}

// Leer ventas ya procesadas
$processed = [];
if (file_exists($processedFile)) {
    $processed = json_decode(file_get_contents($processedFile), true) ?: [];
}

// Leer archivo
$content = file_get_contents($salesFile);
$lines = explode("\n", $content);
$header = null;
$sales = [];

foreach ($lines as $i => $line) {
    $line = trim($line);
    if ($line === '') continue;
    
    $fields = explode("\t", $line);
    
    if ($i === 0 || $header === null) {
        $header = $fields;
        continue;
    }
    
    $sale = [];
    foreach ($header as $idx => $colName) {
        $sale[$colName] = isset($fields[$idx]) ? trim($fields[$idx]) : '';
    }
    
    // Clave única para detectar duplicados
    $saleKey = $sale['ID Venta'] . '_' . $sale['# de Venta'];
    
    if (!isset($processed[$saleKey])) {
        $sales[] = $sale;
    }
}

// Registrar ventas nuevas
$registered = 0;
$skipped = 0;

foreach ($sales as $sale) {
    $idVenta = trim($sale['ID Venta']);
    $numVenta = trim($sale['# de Venta']);
    $saleKey = $idVenta . '_' . $numVenta;
    
    // Parsear datos
    $surtidor = (int)$sale['Surtidor'];
    $manguera = trim($sale['Manguera']);
    $productID = (int)$sale['ID Producto'];
    $volumen = floatval(str_replace(',', '', $sale['Volumen']));
    $monto = floatval(str_replace([',', '$', ' '], '', $sale['Monto']));
    $ppu = floatval(str_replace([',', '$', ' '], '', $sale['PPU']));
    
    // Fecha
    $fechaParts = explode('/', trim($sale['Fecha de Fin']));
    if (count($fechaParts) === 3) {
        $fecha = $fechaParts[2] . '-' . str_pad($fechaParts[1], 2, '0', STR_PAD_LEFT) . '-' . str_pad($fechaParts[0], 2, '0', STR_PAD_LEFT);
    } else {
        $fecha = date('Y-m-d');
    }
    $hora = trim($sale['Hora de Fin']);
    $createdAt = $fecha . ' ' . $hora;
    
    // Buscar manguera
    $stmt = $pdo->prepare("SELECT id FROM hoses WHERE pump_id = ? AND name = ? LIMIT 1");
    $stmt->execute([$surtidor, $manguera]);
    $hose = $stmt->fetch(PDO::FETCH_ASSOC);
    $hoseId = $hose ? (int)$hose['id'] : null;
    
    // Insertar venta
    try {
        $pdo->beginTransaction();
        
        $saleCode = 'S' . $idVenta;
        $stmt = $pdo->prepare("INSERT INTO sales (sale_code, total, payment_method, created_at) VALUES (?, ?, 'cash', ?)");
        $stmt->execute([$saleCode, $monto, $createdAt]);
        $saleId = $pdo->lastInsertId();
        
        // Insertar item
        $stmt = $pdo->prepare("INSERT INTO sale_items (sale_id, product_id, hose_id, qty, unit_price, total) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([$saleId, $productID, $hoseId, $volumen, $ppu, $monto]);
        
        // Actualizar tanques
        $stmt = $pdo->prepare("UPDATE tanques SET vendido = vendido + ?, disponible = disponible - ? WHERE producto_id = ?");
        $stmt->execute([$volumen, $volumen, $productID]);
        
        $pdo->commit();
        
        // Marcar como procesada
        $processed[$saleKey] = time();
        $registered++;
        
    } catch (Exception $e) {
        $pdo->rollBack();
        file_put_contents($logFile, date('[Y-m-d H:i:s] ') . "ERROR: {$saleKey} - {$e->getMessage()}\n", FILE_APPEND);
        $skipped++;
    }
}

// Guardar procesadas
file_put_contents($processedFile, json_encode($processed, JSON_PRETTY_PRINT));

// Log
$msg = "Ventas sincronizadas: {$registered} nuevas, {$skipped} omitidas";
file_put_contents($logFile, date('[Y-m-d H:i:s] ') . $msg . "\n", FILE_APPEND);

echo json_encode(['ok'=>1,'registered'=>$registered,'skipped'=>$skipped,'message'=>$msg]);