<?php
// index.php - Interfaz web mejorada mostrando probabilidades y recomendación
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8" />
  <title>Bitcoin - Precio en tiempo real y tendencia</title>
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <style>
    body { font-family: Arial, Helvetica, sans-serif; margin: 20px; background:#f7f9fc; color:#222; }
    .card { background:white; padding:20px; border-radius:8px; box-shadow:0 2px 8px rgba(0,0,0,0.06); max-width:900px; margin:auto; }
    h1 { margin:0 0 10px 0; }
    .price { font-size:32px; font-weight:700; margin:10px 0; }
    .trend { font-size:18px; margin-left:8px; display:inline-block; vertical-align:middle; }
    .trend.up { color: #14922a; }
    .trend.down { color: #d32f2f; }
    .trend.flat { color: #666; }
    #chart-container { width:100%; height:320px; margin-top:18px; }
    .meta { color:#666; font-size:13px; margin-top:6px; }
    footer { text-align:center; margin-top:18px; color:#999; font-size:13px; }
    .controls { margin-top:10px; display:flex; gap:10px; align-items:center; flex-wrap:wrap; }
    .small { font-size:13px; color:#666; }
    button { padding:6px 10px; border-radius:6px; border:1px solid #ddd; background:white; cursor:pointer; }
    #errorBox { margin-top:12px; padding:10px; border-radius:6px; display:none; color:#7a1f1f; background:#fff0f0; border:1px solid #f1c0c0; }
    #logBox { margin-top:8px; font-family:monospace; font-size:12px; color:#444; max-height:140px; overflow:auto; background:#fafafa; padding:8px; border-radius:6px; border:1px solid #eee; display:none;}
    .probs { display:flex; gap:12px; margin-top:12px; align-items:center; flex-wrap:wrap; }
    .prob-card { padding:12px; border-radius:8px; background:#fafafa; border:1px solid #eee; min-width:180px; text-align:center; }
    .prob-card .big { font-size:22px; font-weight:700; }
    .advice { margin-top:12px; padding:12px; border-radius:8px; font-weight:700; text-align:center; }
    .advice.comprar { background: linear-gradient(90deg,#e8f9ee,#d6f3e0); color:#137a34; border:1px solid #c2e9d0; }
    .advice.vender { background: linear-gradient(90deg,#fff0f0,#ffecec); color:#8b1f1f; border:1px solid #f5c7c7; }
    .advice.mantener { background: linear-gradient(90deg,#fafafa,#f3f3f3); color:#444; border:1px solid #eee; }
  </style>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
  <div class="card">
    <h1>Bitcoin (BTC) - Precio en tiempo real</h1>
    <div>
      <span class="price" id="price">--</span>
      <span id="currency">USD</span>
      <span id="trend" class="trend flat">—</span>
    </div>
    <div class="meta">
      Última actualización: <span id="lastUpdate">--</span>
      &nbsp;•&nbsp;
      Cambio en ventana: <span id="changePct">--</span>
      &nbsp;•&nbsp;
      Muestras: <span id="count">0</span>
    </div>

    <div class="controls">
      <div class="small">Intervalo de actualización:</div>
      <select id="intervalSelect">
        <option value="2000">2s</option>
        <option value="5000" selected>5s</option>
        <option value="10000">10s</option>
        <option value="30000">30s</option>
      </select>
      <div class="small">Ventana de historial (puntos):</div>
      <select id="windowSize">
        <option value="30">30</option>
        <option value="60" selected>60</option>
        <option value="120">120</option>
      </select>
      <button id="btnForce">Forzar actualización</button>
    </div>

    <div id="errorBox"></div>
    <div id="logBox"></div>

    <div id="chart-container">
      <canvas id="priceChart"></canvas>
    </div>

    <div class="probs">
      <div class="prob-card" id="buyCard">
        <div>Probabilidad COMPRAR</div>
        <div class="big" id="buyProb">-- %</div>
      </div>
      <div class="prob-card" id="sellCard">
        <div>Probabilidad VENDER</div>
        <div class="big" id="sellProb">-- %</div>
      </div>
      <div style="flex:1"></div>
    </div>

    <div id="adviceBox" class="advice mantener">--</div>

    <footer>
      Datos por CoinGecko · Aplicación de ejemplo · No es consejo financiero
    </footer>
  </div>

<script>
const apiUrl = 'api/price.php';
let pollInterval = parseInt(document.getElementById('intervalSelect').value, 10);
let timer = null;
let chart = null;
let maxPoints = parseInt(document.getElementById('windowSize').value, 10);

function showError(msg) {
  const el = document.getElementById('errorBox');
  el.style.display = 'block';
  el.textContent = msg;
  const log = document.getElementById('logBox');
  log.style.display = 'block';
  log.textContent = (new Date()).toLocaleTimeString() + ' - ' + msg + '\n' + log.textContent;
  console.error(msg);
}

function clearError() {
  document.getElementById('errorBox').style.display = 'none';
}

function appendLog(msg) {
  const log = document.getElementById('logBox');
  log.style.display = 'block';
  log.textContent = (new Date()).toLocaleTimeString() + ' - ' + msg + '\n' + log.textContent;
  console.log(msg);
}

function formatDate(ts) {
  if (!ts) return '--';
  const d = new Date(ts * 1000);
  return d.toLocaleTimeString();
}

function updateUI(data) {
  clearError();
  document.getElementById('price').textContent = (data.price ? data.price.toFixed(2) : '--');
  document.getElementById('currency').textContent = data.currency || 'USD';
  document.getElementById('lastUpdate').textContent = data.ts ? formatDate(data.ts) : '--';
  document.getElementById('changePct').textContent = data.change_pct !== undefined && data.change_pct !== null ? (data.change_pct > 0 ? '+' : '') + data.change_pct.toFixed(3) + '%' : '--';
  document.getElementById('count').textContent = data.history ? data.history.length : 0;

  const trendEl = document.getElementById('trend');
  trendEl.className = 'trend';
  if (data.trend === 'alcista') { trendEl.textContent = '↑ Alcista'; trendEl.classList.add('up'); }
  else if (data.trend === 'bajista') { trendEl.textContent = '↓ Bajista'; trendEl.classList.add('down'); }
  else { trendEl.textContent = '— Lateral'; trendEl.classList.add('flat'); }

  // Probabilidades y consejo
  if (data.buy_prob !== undefined && data.sell_prob !== undefined) {
    document.getElementById('buyProb').textContent = (data.buy_prob !== null ? data.buy_prob + ' %' : '--');
    document.getElementById('sellProb').textContent = (data.sell_prob !== null ? data.sell_prob + ' %' : '--');
    const adviceBox = document.getElementById('adviceBox');
    adviceBox.textContent = data.advice_text || '--';
    adviceBox.className = 'advice ' + (data.advice || 'mantener');
  }

  if (chart && data.history) {
    const labels = data.history.map(h => formatDate(h.ts));
    const values = data.history.map(h => h.price);
    chart.data.labels = labels;
    chart.data.datasets[0].data = values;
    chart.update();
  }
}

async function fetchPrice() {
  try {
    const params = new URLSearchParams();
    params.set('window', maxPoints);
    const res = await fetch(apiUrl + '?' + params.toString(), {cache: 'no-store'});
    if (!res.ok) {
      const text = await res.text();
      showError('Error HTTP ' + res.status + ': ' + text);
      appendLog('Respuesta no OK: HTTP ' + res.status);
      return;
    }
    const text = await res.text();
    let data = null;
    try {
      data = JSON.parse(text);
    } catch (e) {
      showError('Respuesta inválida JSON desde el servidor: ' + e.message);
      appendLog('Respuesta del servidor (no JSON): ' + text);
      return;
    }

    if (data.error) {
      showError('Error desde API: ' + (data.message || data.error));
      appendLog('API error: ' + JSON.stringify(data));
      // still attempt to update with fallback fields if exist
    }

    updateUI(data);
  } catch (err) {
    showError('Error al obtener datos: ' + err.message);
    appendLog('Fetch exception: ' + err.stack);
  }
}

function startPolling() {
  if (timer) clearInterval(timer);
  fetchPrice();
  timer = setInterval(fetchPrice, pollInterval);
}

document.getElementById('intervalSelect').addEventListener('change', (e) => {
  pollInterval = parseInt(e.target.value, 10);
  startPolling();
});
document.getElementById('windowSize').addEventListener('change', (e) => {
  maxPoints = parseInt(e.target.value, 10);
  fetchPrice();
});
document.getElementById('btnForce').addEventListener('click', fetchPrice);

// Init chart
const ctx = document.getElementById('priceChart').getContext('2d');
chart = new Chart(ctx, {
  type: 'line',
  data: {
    labels: [],
    datasets: [{
      label: 'BTC (USD)',
      data: [],
      borderColor: '#1976d2',
      backgroundColor: 'rgba(25,118,210,0.08)',
      pointRadius: 0,
      borderWidth: 2,
      tension: 0.15,
    }]
  },
  options: {
    responsive: true,
    scales: {
      x: { display: true, ticks: { maxTicksLimit: 10 } },
      y: { display: true }
    },
    plugins: {
      legend: { display: false }
    }
  }
});

// Start
startPolling();
</script>
</body>
</html>