<?php
// api/price.php - versión robusta con fallback, logging y cálculo de probabilidad/advice
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *'); // útil si pruebas desde otro host

// Config
$dbFile = __DIR__ . '/../data/bitcoin.db';
$window = isset($_GET['window']) ? (int)$_GET['window'] : 60; // puntos a devolver
$keepMax = 1000; // máximo de filas a mantener en DB
$trendThresholdPct = 0.1; // umbral en porcentaje para tendencia
$logFile = __DIR__ . '/../data/error.log';
$epsilon = 1e-9; // para evitar divisiones por cero

// Asegurar carpeta data
if (!is_dir(dirname($dbFile))) {
    mkdir(dirname($dbFile), 0755, true);
}

// Helper logging
function logError($msg) {
    global $logFile;
    $time = date('c');
    @file_put_contents($logFile, "[$time] $msg\n", FILE_APPEND | LOCK_EX);
}

// Helper: desviación estándar
function stddev(array $a) {
    $n = count($a);
    if ($n <= 1) return 0.0;
    $mean = array_sum($a) / $n;
    $sum = 0.0;
    foreach ($a as $v) $sum += ($v - $mean) * ($v - $mean);
    return sqrt($sum / ($n - 1));
}

// Sigmoid
function sigmoid($x) {
    return 1.0 / (1.0 + exp(-$x));
}

try {
    $pdo = new PDO('sqlite:' . $dbFile);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Crear tabla si no existe
    $pdo->exec("CREATE TABLE IF NOT EXISTS prices (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ts INTEGER NOT NULL,
        price REAL NOT NULL
    );");
    $pdo->exec("CREATE INDEX IF NOT EXISTS idx_ts ON prices(ts);");

    // 1) Consultar CoinGecko con cURL primero, fallback a file_get_contents
    $cgUrl = 'https://api.coingecko.com/api/v3/simple/price?ids=bitcoin&vs_currencies=usd';
    $price = null;
    $ts = time();

    // Intent cURL
    if (function_exists('curl_init')) {
        $ch = curl_init($cgUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_USERAGENT, 'PHP-BTC-Price-App/1.0');
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        $resp = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlErr = curl_error($ch);
        curl_close($ch);

        if ($resp !== false && $httpCode < 400) {
            $json = json_decode($resp, true);
            if (isset($json['bitcoin']['usd'])) {
                $price = floatval($json['bitcoin']['usd']);
            } else {
                logError("cURL response missing bitcoin.usd: " . substr($resp, 0, 400));
            }
        } else {
            logError("cURL failed HTTP=$httpCode err='$curlErr' resp=" . substr((string)$resp,0,400));
        }
    }

    // Fallback file_get_contents
    if ($price === null && ini_get('allow_url_fopen')) {
        $opts = ['http'=>['method'=>'GET','header'=>"User-Agent: PHP-BTC-Price-App/1.0\r\n"], 'ssl'=>['verify_peer'=>true,'verify_peer_name'=>true]];
        $context = stream_context_create($opts);
        $resp = @file_get_contents($cgUrl, false, $context);
        if ($resp !== false) {
            $json = json_decode($resp, true);
            if (isset($json['bitcoin']['usd'])) {
                $price = floatval($json['bitcoin']['usd']);
            } else {
                logError("file_get_contents response missing bitcoin.usd: " . substr($resp,0,400));
            }
        } else {
            logError("file_get_contents failed or blocked (allow_url_fopen?)");
        }
    }

    // Si obtuvimos precio -> insertar y devolver normal
    if ($price !== null) {
        $stmt = $pdo->prepare('INSERT INTO prices(ts, price) VALUES(:ts, :price)');
        $stmt->execute([':ts' => $ts, ':price' => $price]);
        // Mantener solo últimas keepMax filas
        $pdo->exec("DELETE FROM prices WHERE id NOT IN (SELECT id FROM prices ORDER BY ts DESC LIMIT $keepMax)");
    } else {
        // No pudimos obtener precio: registrar y vamos a devolver historial (fallback)
        logError("No se pudo obtener precio desde CoinGecko en " . date('c'));
        respondWithHistory($pdo, $window, "error_api");
        exit;
    }

    // Obtener historial
    $stmt = $pdo->prepare('SELECT ts, price FROM prices ORDER BY ts DESC LIMIT :lim');
    $stmt->bindValue(':lim', $window, PDO::PARAM_INT);
    $stmt->execute();
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $rows = array_reverse($rows);

    // Calcular tendencia
    $historyCount = count($rows);
    $changePct = null;
    $trend = 'lateral';
    if ($historyCount >= 2) {
        $first = $rows[0]['price'];
        $last = $rows[$historyCount - 1]['price'];
        if ($first != 0.0) {
            $changePct = (($last - $first) / $first) * 100.0; // en %
        }
        if ($changePct > $trendThresholdPct) $trend = 'alcista';
        else if ($changePct < -$trendThresholdPct) $trend = 'bajista';
        else $trend = 'lateral';
    }

    // Calcular volatilidad (stddev de retornos) en %
    $vol_pct = 0.0;
    if ($historyCount >= 2) {
        $returns = [];
        for ($i = 1; $i < $historyCount; $i++) {
            $p0 = $rows[$i-1]['price'];
            $p1 = $rows[$i]['price'];
            if ($p0 > 0) $returns[] = ($p1 - $p0) / $p0;
        }
        $vol = stddev($returns); // en unidades decimales
        $vol_pct = $vol * 100.0; // pasar a %
    }

    // Score y probabilidades
    // score = changePct / (vol_pct + small), luego sigmoid(k * score)
    $k = 0.7; // factor de sensibilidad
    $score = 0.0;
    if ($changePct !== null) {
        $score = $changePct / (max($vol_pct, $epsilon) + $epsilon);
    }
    $prob_buy = sigmoid($k * $score); // 0..1
    $prob_buy_pct = round($prob_buy * 100.0, 1);
    $prob_sell_pct = round((1.0 - $prob_buy) * 100.0, 1);

    // Advice simple:
    $advice = 'mantener';
    $advice_text = 'Mantener / Esperar';
    if ($prob_buy_pct >= 60.0) {
        $advice = 'comprar';
        $advice_text = 'Deberías considerar COMPRAR';
    } else if ($prob_sell_pct >= 60.0) {
        $advice = 'vender';
        $advice_text = 'Deberías considerar VENDER';
    }

    $out = [
        'price' => $price,
        'currency' => 'USD',
        'ts' => $ts,
        'trend' => $trend,
        'change_pct' => $changePct,
        'history' => array_map(function($r){ return ['ts'=> (int)$r['ts'], 'price'=> (float)$r['price']]; }, $rows),
        // Nuevos campos para probabilidades y consejo
        'buy_prob' => $prob_buy_pct,
        'sell_prob' => $prob_sell_pct,
        'volatility_pct' => round($vol_pct, 4),
        'score' => round($score, 6),
        'advice' => $advice,
        'advice_text' => $advice_text,
    ];

    echo json_encode($out, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;

} catch (Exception $e) {
    $msg = $e->getMessage();
    logError("Exception in api/price.php: $msg");
    http_response_code(500);
    echo json_encode(['error' => 'server_error', 'message' => $msg]);
    exit;
}

function respondWithHistory($pdo, $window, $reason) {
    try {
        $stmt = $pdo->prepare('SELECT ts, price FROM prices ORDER BY ts DESC LIMIT :lim');
        $stmt->bindValue(':lim', $window, PDO::PARAM_INT);
        $stmt->execute();
        $rows = array_reverse($stmt->fetchAll(PDO::FETCH_ASSOC));
        $historyCount = count($rows);
        $changePct = null;
        $trend = 'lateral';
        if ($historyCount >= 2) {
            $first = $rows[0]['price'];
            $last = $rows[$historyCount - 1]['price'];
            if ($first != 0.0) {
                $changePct = (($last - $first) / $first) * 100.0;
            }
            $trend = ($changePct > 0.1) ? 'alcista' : (($changePct < -0.1) ? 'bajista' : 'lateral');
        }

        // Calcular volatilidad y probabilidades incluso en fallback
        $vol_pct = 0.0;
        if ($historyCount >= 2) {
            $returns = [];
            for ($i = 1; $i < $historyCount; $i++) {
                $p0 = $rows[$i-1]['price'];
                $p1 = $rows[$i]['price'];
                if ($p0 > 0) $returns[] = ($p1 - $p0) / $p0;
            }
            $vol = stddev($returns);
            $vol_pct = $vol * 100.0;
        }
        $k = 0.7;
        $epsilon = 1e-9;
        $score = ($changePct !== null) ? ($changePct / (max($vol_pct, $epsilon) + $epsilon)) : 0.0;
        $prob_buy_pct = round(sigmoid($k * $score) * 100.0, 1);
        $prob_sell_pct = round((1.0 - sigmoid($k * $score)) * 100.0, 1);

        $advice = 'mantener';
        $advice_text = 'Mantener / Esperar';
        if ($prob_buy_pct >= 60.0) { $advice = 'comprar'; $advice_text = 'Deberías considerar COMPRAR'; }
        else if ($prob_sell_pct >= 60.0) { $advice = 'vender'; $advice_text = 'Deberías considerar VENDER'; }

        $out = [
            'price' => ($historyCount ? (float)$rows[$historyCount-1]['price'] : null),
            'currency' => 'USD',
            'ts' => ($historyCount ? (int)$rows[$historyCount-1]['ts'] : null),
            'trend' => $trend,
            'change_pct' => $changePct,
            'history' => array_map(function($r){ return ['ts'=> (int)$r['ts'], 'price'=> (float)$r['price']]; }, $rows),
            'buy_prob' => $prob_buy_pct,
            'sell_prob' => $prob_sell_pct,
            'volatility_pct' => round($vol_pct, 4),
            'score' => round($score, 6),
            'advice' => $advice,
            'advice_text' => $advice_text,
            'note' => 'fallback_due_to_'.$reason,
        ];
        echo json_encode($out, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    } catch (Exception $e) {
        logError("respondWithHistory failed: " . $e->getMessage());
        echo json_encode(['error' => 'fallback_failed', 'message' => $e->getMessage()]);
    }
    exit;
}